/**
 *-----------------------------------------------------------------------------------
 *    Filename: ADFImplicitFixed.c
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Copyright 2008 Monotype Imaging Inc.
 *    Copyright 2004-2007 Mitsubishi Electric Research Laboratories (MERL)
 *    An implementation for processing (e.g., generating and rendering) implicit ADFs
 *    Sarah Frisken and Ronald Perry
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    This is a fixed point implementation for processing implicit ADFs and is modelled
 *    after the floating point implementation contained in ADFImplicitFloat.c.
 *    Deviations from the floating point implementation as well as fixed point
 *    implementation details (e.g., the use of lookup tables instead of direct
 *    evaluation) are annotated as fixed point math notes, i.e., FixedMathNotes.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    The original implementation for processing implicit ADFs allowed overlapping
 *    contour boundaries to be shaded gray regardless of whether the boundary sample
 *    point was actually on a boundary or inside the glyph. This created objectionable 
 *    "missing" pixels within the interior of the glyph. While a completely proper fix
 *    for this situation is not possible without a complete redesign of the algorithm,
 *    this version implements a modification of the original algorithm that produces 
 *    improved results for overlapping closed loop contours. Uniform stroke paths
 *    function properly using the original design and that processing was retained in
 *    this version. 
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Throughout this file, suggestions for possible implementation changes are
 *    annotated as development notes, i.e., DevNotes.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Overview
 *
 *        This file contains the following major functional blocks: 
 *
 *            (1) Implicit ADF generation
 *            (2) Implicit ADF rendering
 *
 *        Implicit ADF generation converts a given ADFPath from font units to ADF
 *        coordinates. The resultant representation (viewed as an opaque ADF to the 
 *        application but considered a preprocessed ADFPath internally) can effectively 
 *        be scaled, translated, rotated, sheared, etc. (by setting the appropriate 
 *        ADFRenderSetup() attributes) prior to rendering without affecting the quality 
 *        of the rendered image. Note also that the preprocessed ADFPath (i.e., the 
 *        generated ADF) can be cached as an ADF using the library's dual caching 
 *        system.
 *
 *        Implicit ADF rendering generates an implicit ADF from a given preprocessed
 *        ADFPath (viewed as an opaque ADF to the application), renders the implicit
 *        ADF into a distance buffer using the specified rendering attributes, and then
 *        maps distances in the distance buffer to density values. These density values
 *        are packed into pixels of a specified density image. Implicit ADF rendering
 *        of an outline-based glyph operates differently than implicit ADF rendering of
 *        a uniform-width stroke-based glyph. Implicit ADF rendering of an
 *        outline-based glyph comprises the following steps:
 *
 *            (1) Each pixel (for CRT rendering) or each pixel component (for LCD
 *                rendering) of the specified density image is cleared to zero.
 *
 *            (2) A distance buffer is created large enough to store a distance sample
 *                for each pixel or pixel component of the specified density image. A
 *                distance buffer is a 2D array of ADF_I1616 fixed point values that is
 *                used to combine distances from contributing elements (i.e., line
 *                segments and corners) of the preprocessed ADFPath at each sample
 *                point. Each distance sample in the distance buffer is initialized to
 *                LARGE_OUTSIDE_DIST_VAL. Note that a distance value of
 *                LARGE_OUTSIDE_DIST_VAL maps to zero density in the density image.
 *
 *            (3) A winding number buffer is created large enough to store the winding
 *                number for each pixel of the corresponding density image. This buffer
 *                is a 2D array of ADF_I8 values that represent the absolute value of the
 *                actual winding number for each pixel in the image. This number is the
 *                number of contours which contain the sample point. Using this winding
 *                number buffer, all distance samples which are inside a contour 
 *                (i.e. have a winding number > 0) are set to LARGE_INSIDE_DIST_VAL. This
 *                value maps to maximum density in the density image. 
 *
 *            (4) A boundary cell buffer is also created large enough to store distance
 *                samples for each pixel in the corresponding density image. These
 *                distances are initialized to zero. If a boundary cell sample point is 
 *                covered by two or more contours (i.e. its corresponding winding number
 *                buffer value is > 1), then the boundary cell buffer is used to 
 *                temporarily store distance values for the multiple boundaries. The 
 *                first boundary that processes the sample point sets the boundary cell
 *                value to be the negative of the distance to indicate that it is the
 *                first value. Subsequent boundaries that affect that sample point will
 *                set the distance value in the distance buffer to be the largest of the
 *                distance values in the boundary cell buffer. If the winding number for
 *                the sample point is > 1 but only one boundary processes that sample
 *                point, the distance value is left as LARGE_INSIDE_DIST_VAL. Corner
 *                sample points which lie near the end points of horizontal and vertical
 *                lines are considered to be a single instance of a boundary point.
 *        
 *            (5) Pen commands in the preprocessed ADFPath are transformed from ADF
 *                coordinates to ADF_I1616 fixed point image coordinates. If requested,
 *                MAZ alignment zone detection and grid fitting are performed on the
 *                transformed pen commands (see ADFAlgnZonesMAZ.h,
 *                ADFAlgnZonesMAZOutlines.c, and ADFAlgnZonesMAZStrokes.c for details).
 *
 *            (6) An internal path is created from the transformed pen commands. The
 *                internal path is used to generate and render implicit ADF boundary
 *                cells and to rasterize the glyph interior. Curvto commands in the
 *                transformed pen commands are not added to the internal path, but
 *                instead are replaced by a sequence of lineto commands that closely
 *                approximate the corresponding curve segment. Subdividing curve
 *                segments into line segments has significant advantages in performance
 *                and ease of implementation over processing curve segments directly
 *                (see CreateInternPath() for more details).
 *    
 *            (7) Each element of the internal path is processed. For each element, an
 *                implicit ADF boundary cell is generated. The implicit ADF boundary
 *                cell includes a representation of the cell's geometry (e.g., its
 *                vertices) and data required to compute the minimum unsigned Euclidean
 *                distance from any point inside the implicit ADF boundary cell to the
 *                element represented by the implicit ADF boundary cell. Implicit ADF
 *                boundary cells are represented in ADF_I1616 fixed point image
 *                coordinates. The size and geometry of an implicit ADF boundary cell
 *                is constructed such that it covers the area obtained by sweeping a
 *                line segment along the section of the internal path corresponding to
 *                the element represented by the implicit ADF boundary cell, where the
 *                swept line segment is perpendicular to the element and extends on
 *                each side of the element by the specified CSM filter cutoff values.
 *                Consequently, the union of the geometry of the implicit ADF boundary
 *                cells for all elements of the internal path is guaranteed to cover
 *                those samples points which require antialiasing. By computing
 *                distances only near the edge of a glyph, the time required to render
 *                the glyph is minimized. As each implicit ADF boundary cell is
 *                generated, the implicit ADF boundary cell is rendered into the
 *                distance buffer by rasterizing the cell interior, determining the
 *                minimum unsigned Euclidean distance from each sample point to the
 *                element represented by the implicit ADF boundary cell, and combining
 *                the determined distance value with the corresponding distance sample
 *                stored in the distance buffer. 
 *
 *            (8) Distance values exterior to the glyph are considered to be negative 
 *                while interior distances are considered to be positive. The combining
 *                process selects the minimum magnitude distance. 
 *
 *            (9) Distances in the distance buffer are mapped to density values by
 *                applying the given CSM parameters. Color reduction is applied to
 *                these density values during LCD rendering if enabled. The final
 *                density values are packed into the specified ADFImage, where the
 *                packing depends on the display mode.
 *
 *        Implicit ADF rendering of a uniform-width stroke-based glyph differs from
 *        implicit ADF rendering of an outline-based glyph in the following ways:
 *
 *            (a) Before the elements of the internal path are processed as described
 *                above, the CSM inside and outside filter cutoff values
 *                are both decreased by half the stroke width. These adjustments are
 *                required to account for the stroke width of the uniform-width
 *                stroke-based glyph. Effectively, this step moves the filter position
 *                outward (i.e., away from the centerlines of the glyph) by half the
 *                stroke width without changing the CSM filter width.
 *
 *            (b) The winding number buffer and boundary cell buffer are not used and
 *                do not apply to open loop uniform stroke paths. NULL pointers for these
 *                buffers are passed to the processing routines to indicate that they 
 *                do not apply. 
 *
 *            (c) Setting the interior of the glyph to LARGE_INSIDE_DIST_VAL is not
 *                performed. In the case of a uniform-width stroke-based
 *                glyph, the interior is rendered during step (7) (i.e., during the
 *                processing of each element of the internal path). Therefore, it is
 *                unnecessary to rasterize the interior of the implicit ADF glyph in a
 *                separate step. All distance values are treated as negative in this 
 *                and are combined to select the minimum absolute distance.
 *
 *            (c) After the distances in the distance buffer have been mapped to
 *                density values as described above in step (7), the adjusted CSM
 *                inside and outside filter cutoff values are restored to their
 *                original values.
 *
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    DevNote: Insufficient memory can cause things to fail when processing (e.g.,
 *    rendering) a glyph -- e.g., when creating an internal path or a distance buffer.
 *    Memory allocation can also be slow, particularly if allocations are required on a
 *    per-glyph basis. Furthermore, the memory requirements for some data structures
 *    could be large (e.g., a distance buffer for a full screen LCD image). Consider
 *    redesigning the memory allocation strategy to eliminate failures and to provide
 *    optimal performance (e.g., the need for an explicitly allocated distance buffer
 *    may not be necessary).
 *-----------------------------------------------------------------------------------
 */


/**
 *-----------------------------------------------------------------------------------
 *    Required include files for this implementation
 *-----------------------------------------------------------------------------------
 */
#include <math.h>
#include <string.h>
/*#include <stdlib.h>*/ /* not needed per Lint */

#include "fs_object.h"
#include "fs_function.h"

/**
 *-----------------------------------------------------------------------------------
 *    START: iType Edge Rendering
 *-----------------------------------------------------------------------------------
 */
#ifdef FS_EDGE_RENDER

#define SIN_2_DEGREES 2287
#define MYINFINITY  2147483647L /* fs_fixed.h */
#define COS_CHECK 0x400 /* 1024 */


#include "adffixedmath.h"
#include "adfinittermsystem.h"
#include "adfgenerate.h"
#include "adfimplicit.h"
#include "adfalgnzonesmaz.h"



/**
 *-------------------------------------------------------------------------------
 *    START: IMPLICIT ADFS ONLY
 *-------------------------------------------------------------------------------
 */
#if (ADF_USE_IMPLICIT_ADFS)


/**
 *-------------------------------------------------------------------------------
 *    START: FIXED POINT MATH ONLY
 *-------------------------------------------------------------------------------
 */

/**
 *-----------------------------------------------------------------------------------
 *    RECIPROCAL LOOKUP TABLE
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    rcpTable[] is a table that maps an integer i to its reciprocal (i.e., 1/i). Each
 *    element of the table is stored as an ADF_I1616 fixed point value. The table
 *    contains 512 elements and is precomputed as follows:
 *
 *    rcpTable[i] = ((i > 0) ? floor(65536.0 / i) : 0), where i is an integer that lies
 *    in the range [0, 511]. Note that excess fractional bits are discarded (i.e., each
 *    element is rounded towards zero).
 *
 *    The first element of the table (i.e., the element that corresponds to the
 *    undefined reciprocal of zero) is arbitrarily set to zero.
 *-----------------------------------------------------------------------------------
 */
static FS_CONST ADF_I1616 rcpTable[] = {
    0x00000, 0x10000, 0x08000, 0x05555, 0x04000, 0x03333, 0x02aaa, 0x02492, 
    0x02000, 0x01c71, 0x01999, 0x01745, 0x01555, 0x013b1, 0x01249, 0x01111, 
    0x01000, 0x00f0f, 0x00e38, 0x00d79, 0x00ccc, 0x00c30, 0x00ba2, 0x00b21, 
    0x00aaa, 0x00a3d, 0x009d8, 0x0097b, 0x00924, 0x008d3, 0x00888, 0x00842, 
    0x00800, 0x007c1, 0x00787, 0x00750, 0x0071c, 0x006eb, 0x006bc, 0x00690, 
    0x00666, 0x0063e, 0x00618, 0x005f4, 0x005d1, 0x005b0, 0x00590, 0x00572, 
    0x00555, 0x00539, 0x0051e, 0x00505, 0x004ec, 0x004d4, 0x004bd, 0x004a7, 
    0x00492, 0x0047d, 0x00469, 0x00456, 0x00444, 0x00432, 0x00421, 0x00410, 
    0x00400, 0x003f0, 0x003e0, 0x003d2, 0x003c3, 0x003b5, 0x003a8, 0x0039b, 
    0x0038e, 0x00381, 0x00375, 0x00369, 0x0035e, 0x00353, 0x00348, 0x0033d, 
    0x00333, 0x00329, 0x0031f, 0x00315, 0x0030c, 0x00303, 0x002fa, 0x002f1, 
    0x002e8, 0x002e0, 0x002d8, 0x002d0, 0x002c8, 0x002c0, 0x002b9, 0x002b1, 
    0x002aa, 0x002a3, 0x0029c, 0x00295, 0x0028f, 0x00288, 0x00282, 0x0027c, 
    0x00276, 0x00270, 0x0026a, 0x00264, 0x0025e, 0x00259, 0x00253, 0x0024e, 
    0x00249, 0x00243, 0x0023e, 0x00239, 0x00234, 0x00230, 0x0022b, 0x00226, 
    0x00222, 0x0021d, 0x00219, 0x00214, 0x00210, 0x0020c, 0x00208, 0x00204, 
    0x00200, 0x001fc, 0x001f8, 0x001f4, 0x001f0, 0x001ec, 0x001e9, 0x001e5, 
    0x001e1, 0x001de, 0x001da, 0x001d7, 0x001d4, 0x001d0, 0x001cd, 0x001ca, 
    0x001c7, 0x001c3, 0x001c0, 0x001bd, 0x001ba, 0x001b7, 0x001b4, 0x001b2, 
    0x001af, 0x001ac, 0x001a9, 0x001a6, 0x001a4, 0x001a1, 0x0019e, 0x0019c, 
    0x00199, 0x00197, 0x00194, 0x00192, 0x0018f, 0x0018d, 0x0018a, 0x00188, 
    0x00186, 0x00183, 0x00181, 0x0017f, 0x0017d, 0x0017a, 0x00178, 0x00176, 
    0x00174, 0x00172, 0x00170, 0x0016e, 0x0016c, 0x0016a, 0x00168, 0x00166, 
    0x00164, 0x00162, 0x00160, 0x0015e, 0x0015c, 0x0015a, 0x00158, 0x00157, 
    0x00155, 0x00153, 0x00151, 0x00150, 0x0014e, 0x0014c, 0x0014a, 0x00149, 
    0x00147, 0x00146, 0x00144, 0x00142, 0x00141, 0x0013f, 0x0013e, 0x0013c, 
    0x0013b, 0x00139, 0x00138, 0x00136, 0x00135, 0x00133, 0x00132, 0x00130, 
    0x0012f, 0x0012e, 0x0012c, 0x0012b, 0x00129, 0x00128, 0x00127, 0x00125, 
    0x00124, 0x00123, 0x00121, 0x00120, 0x0011f, 0x0011e, 0x0011c, 0x0011b, 
    0x0011a, 0x00119, 0x00118, 0x00116, 0x00115, 0x00114, 0x00113, 0x00112, 
    0x00111, 0x0010f, 0x0010e, 0x0010d, 0x0010c, 0x0010b, 0x0010a, 0x00109, 
    0x00108, 0x00107, 0x00106, 0x00105, 0x00104, 0x00103, 0x00102, 0x00101, 
    0x00100, 0x000ff, 0x000fe, 0x000fd, 0x000fc, 0x000fb, 0x000fa, 0x000f9, 
    0x000f8, 0x000f7, 0x000f6, 0x000f5, 0x000f4, 0x000f3, 0x000f2, 0x000f1, 
    0x000f0, 0x000f0, 0x000ef, 0x000ee, 0x000ed, 0x000ec, 0x000eb, 0x000ea, 
    0x000ea, 0x000e9, 0x000e8, 0x000e7, 0x000e6, 0x000e5, 0x000e5, 0x000e4, 
    0x000e3, 0x000e2, 0x000e1, 0x000e1, 0x000e0, 0x000df, 0x000de, 0x000de, 
    0x000dd, 0x000dc, 0x000db, 0x000db, 0x000da, 0x000d9, 0x000d9, 0x000d8, 
    0x000d7, 0x000d6, 0x000d6, 0x000d5, 0x000d4, 0x000d4, 0x000d3, 0x000d2, 
    0x000d2, 0x000d1, 0x000d0, 0x000d0, 0x000cf, 0x000ce, 0x000ce, 0x000cd, 
    0x000cc, 0x000cc, 0x000cb, 0x000ca, 0x000ca, 0x000c9, 0x000c9, 0x000c8, 
    0x000c7, 0x000c7, 0x000c6, 0x000c5, 0x000c5, 0x000c4, 0x000c4, 0x000c3, 
    0x000c3, 0x000c2, 0x000c1, 0x000c1, 0x000c0, 0x000c0, 0x000bf, 0x000bf, 
    0x000be, 0x000bd, 0x000bd, 0x000bc, 0x000bc, 0x000bb, 0x000bb, 0x000ba, 
    0x000ba, 0x000b9, 0x000b9, 0x000b8, 0x000b8, 0x000b7, 0x000b7, 0x000b6, 
    0x000b6, 0x000b5, 0x000b5, 0x000b4, 0x000b4, 0x000b3, 0x000b3, 0x000b2, 
    0x000b2, 0x000b1, 0x000b1, 0x000b0, 0x000b0, 0x000af, 0x000af, 0x000ae, 
    0x000ae, 0x000ad, 0x000ad, 0x000ac, 0x000ac, 0x000ac, 0x000ab, 0x000ab, 
    0x000aa, 0x000aa, 0x000a9, 0x000a9, 0x000a8, 0x000a8, 0x000a8, 0x000a7, 
    0x000a7, 0x000a6, 0x000a6, 0x000a5, 0x000a5, 0x000a5, 0x000a4, 0x000a4, 
    0x000a3, 0x000a3, 0x000a3, 0x000a2, 0x000a2, 0x000a1, 0x000a1, 0x000a1, 
    0x000a0, 0x000a0, 0x0009f, 0x0009f, 0x0009f, 0x0009e, 0x0009e, 0x0009d, 
    0x0009d, 0x0009d, 0x0009c, 0x0009c, 0x0009c, 0x0009b, 0x0009b, 0x0009a, 
    0x0009a, 0x0009a, 0x00099, 0x00099, 0x00099, 0x00098, 0x00098, 0x00098, 
    0x00097, 0x00097, 0x00097, 0x00096, 0x00096, 0x00095, 0x00095, 0x00095, 
    0x00094, 0x00094, 0x00094, 0x00093, 0x00093, 0x00093, 0x00092, 0x00092, 
    0x00092, 0x00091, 0x00091, 0x00091, 0x00090, 0x00090, 0x00090, 0x00090, 
    0x0008f, 0x0008f, 0x0008f, 0x0008e, 0x0008e, 0x0008e, 0x0008d, 0x0008d, 
    0x0008d, 0x0008c, 0x0008c, 0x0008c, 0x0008c, 0x0008b, 0x0008b, 0x0008b, 
    0x0008a, 0x0008a, 0x0008a, 0x00089, 0x00089, 0x00089, 0x00089, 0x00088, 
    0x00088, 0x00088, 0x00087, 0x00087, 0x00087, 0x00087, 0x00086, 0x00086, 
    0x00086, 0x00086, 0x00085, 0x00085, 0x00085, 0x00084, 0x00084, 0x00084, 
    0x00084, 0x00083, 0x00083, 0x00083, 0x00083, 0x00082, 0x00082, 0x00082, 
    0x00082, 0x00081, 0x00081, 0x00081, 0x00081, 0x00080, 0x00080, 0x00080, 
};

/**
 *-----------------------------------------------------------------------------------
 *    IMPLICIT ADF GENERATION
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Generate an implicit ADF representing the glyph described by the specified 
 *    ADFPath. A pointer to the implicit ADF is returned if the implicit ADF is
 *    generated successfully; a NULL is returned if the request cannot be satisfied.
 *
 *    Implicit ADF generation transforms pen commands of an ADFPath from font units to 
 *    the [0.0,1.0] x [0.0,1.0] ADF coordinate system and stores the transformed pen 
 *    commands in a contiguous array immediately following the ADF header (i.e., the 
 *    ADFGlyph data structure). ADFGlyph.numCells is used to store the number of 
 *    transformed pen commands and ADFGlyph.rootOffset is used to store the offset in 
 *    bytes from the start of the ADF header to the array of transformed pen commands.
 *    Note that implicit ADF cells are not generated during generation -- they are 
 *    generated on-demand during rendering.
 *-----------------------------------------------------------------------------------
 */
ADFGlyph *ADFGenerateADFImplicit (void *libInst, ADFPath *path)
{
    ADF_U32   n;
    ADF_U32   totalSize;
    ADF_U32   glyphHdrSize;
    ADF_U32   numPenCmds;
    ADF_I1616 FUToADFScale; 
    ADF_I1616 adfGlyphOriginX;
    ADF_I1616 adfGlyphOriginY;
    ADF_I1616 pathGlyphMinX;
    ADF_I1616 pathGlyphMinY;
    ADF_I1616 pathGlyphMaxX;
    ADF_I1616 pathGlyphMaxY;
    ADF_I1616 adfGlyphMinX;
    ADF_I1616 adfGlyphMinY;
    ADF_I1616 adfGlyphMaxX;
    ADF_I1616 adfGlyphMaxY;
    ADFPenCmd *pathPenCmd;
    ADFPenCmd *adfPenCmd;
    ADFGlyph  *adf;


    /**
     *----appInst is required for all memory allocation tasks
     */
    void *appInst = ADFGetAppInst(libInst);


    /**
     *----Return NULL if the ADFPath is invalid
     */
    if (path == 0) return(0);


    /**
     *----Determine the size of the implicit ADF and the offset in bytes from the 
     *----start of the ADF header (i.e., the ADFGlyph) to the array of transformed 
     *----pen commands
     */
    glyphHdrSize = sizeof(ADFGlyph);
    numPenCmds = path->numPenCmds;
    totalSize = glyphHdrSize + numPenCmds * sizeof(ADFPenCmd);


    /**
     *----Allocate the implicit ADFGlyph and set its size and the offset from the 
     *----start of the ADF header to the array of transformed pen commands. Return 
     *----a NULL ADF if the allocation fails.
     */
#ifdef FS_MEM_DBG
    ((FS_STATE *)appInst)->memdbgid = "ADFGlyph";
#endif
    if ((adf = (ADFGlyph *) ADF_ALLOC(appInst, totalSize)) == NULL)
    {
        ((FS_STATE *)appInst)->error = ERR_MALLOC_FAIL;
        return(0);
    }
    adf->totalSizeBytes = totalSize;
    adf->rootOffset = glyphHdrSize;


    /**
     *----Set various known attributes of the implicit ADFGlyph
     */
    adf->numCells = numPenCmds;
    adf->charCode = path->charCode;
    adf->version = ADF_IMPLICIT_VERSION_NUMBER;


    /**
     *----Determine the scale and offset for transforming the glyph represented by 
     *----the ADFPath from font units to the [0.0,1.0] x [0.0,1.0] ADF coordinate 
     *----system and compute the EM box size of the typeface of the glyph in ADF 
     *----units. Store these values in the ADF header.
     */
    ADFSetGlyphScaleAndOffset(path, &adf->FUToADFScale, &adf->glyphOriginX, 
    &adf->glyphOriginY, &adf->ADFUnitsPerEM);


    /**
     *----Convert the scale and offset from floating point values to ADF_I1616 fixed
     *----point values
     */
    FUToADFScale    = FLOAT_TO_I1616(adf->FUToADFScale);
    adfGlyphOriginX = FLOAT_TO_I1616(adf->glyphOriginX);
    adfGlyphOriginY = FLOAT_TO_I1616(adf->glyphOriginY);


    /**
     *----Convert the ADFPath's bounding box from floating point values to ADF_I1616
     *----fixed point values
     */
    pathGlyphMinX = FLOAT_TO_I1616(path->glyphMinX);
    pathGlyphMinY = FLOAT_TO_I1616(path->glyphMinY);
    pathGlyphMaxX = FLOAT_TO_I1616(path->glyphMaxX);
    pathGlyphMaxY = FLOAT_TO_I1616(path->glyphMaxY);


    /**
     *----Determine the exact bounding box (in ADF coordinates) of the glyph 
     *----represented by the ADFPath. 
     */
    adfGlyphMinX = I1616_MUL(pathGlyphMinX, FUToADFScale) + adfGlyphOriginX;
    adfGlyphMinY = I1616_MUL(pathGlyphMinY, FUToADFScale) + adfGlyphOriginY;
    adfGlyphMaxX = I1616_MUL(pathGlyphMaxX, FUToADFScale) + adfGlyphOriginX;
    adfGlyphMaxY = I1616_MUL(pathGlyphMaxY, FUToADFScale) + adfGlyphOriginY;


    /**
     *----Set the path type and stroke width in the ADF header. For uniform-width 
     *----stroke-based glyphs, convert the stroke width into ADF coordinates and 
     *----correct the exact bounding box of the glyph to account for the stroke 
     *----width.
     */
    adf->pathType = path->pathType;
    adf->pathWidth = 0 /*0.0f*/;
    if (adf->pathType == ADF_UNIFORM_STROKE_PATH) {
        ADF_I1616 strokeWidth = I1616_MUL(
        FLOAT_TO_I1616(path->pathWidth), FUToADFScale);
        ADF_I1616 strokeRadius = strokeWidth >> 1;
        adf->pathWidth = I1616_TO_FLOAT(strokeWidth);
        adfGlyphMinX -= strokeRadius;
        adfGlyphMinY -= strokeRadius;
        adfGlyphMaxX += strokeRadius;
        adfGlyphMaxY += strokeRadius;
    }


    /**
     *----Transform each pen command from font units to the [0.0,1.0] x [0.0,1.0] ADF
     *----coordinate system and copy the transformed pen command into the implicit
     *----ADF
     */
    pathPenCmd = path->penCmds;
    adfPenCmd = (ADFPenCmd *) (((ADF_U8 *) adf) + adf->rootOffset);
    for (n = 0; n < numPenCmds; n++, pathPenCmd++, adfPenCmd++) {
        adfPenCmd->opCode = pathPenCmd->opCode;
        adfPenCmd->x = I1616_TO_FLOAT(I1616_MUL(FLOAT_TO_I1616(
        pathPenCmd->x), FUToADFScale) + adfGlyphOriginX);
        adfPenCmd->y = I1616_TO_FLOAT(I1616_MUL(FLOAT_TO_I1616(
        pathPenCmd->y), FUToADFScale) + adfGlyphOriginY);
        if (adfPenCmd->opCode == ADF_PEN_CURVTO_CMD) {
            adfPenCmd->cx = I1616_TO_FLOAT(I1616_MUL(FLOAT_TO_I1616(
            pathPenCmd->cx), FUToADFScale) + adfGlyphOriginX);
            adfPenCmd->cy = I1616_TO_FLOAT(I1616_MUL(FLOAT_TO_I1616(
            pathPenCmd->cy), FUToADFScale) + adfGlyphOriginY);
        }
        if (adfPenCmd->opCode == ADF_PEN_CUBETO_CMD) {
            adfPenCmd->cx = I1616_TO_FLOAT(I1616_MUL(FLOAT_TO_I1616(
            pathPenCmd->cx), FUToADFScale) + adfGlyphOriginX);
            adfPenCmd->cy = I1616_TO_FLOAT(I1616_MUL(FLOAT_TO_I1616(
            pathPenCmd->cy), FUToADFScale) + adfGlyphOriginY);
            adfPenCmd->cx2 = I1616_TO_FLOAT(I1616_MUL(FLOAT_TO_I1616(
            pathPenCmd->cx2), FUToADFScale) + adfGlyphOriginX);
            adfPenCmd->cy2 = I1616_TO_FLOAT(I1616_MUL(FLOAT_TO_I1616(
            pathPenCmd->cy2), FUToADFScale) + adfGlyphOriginY);
        }
    }


    /**
     *----Convert the bounding box from ADF_I1616 fixed point values to floating
     *----point values and store the results in the ADF header
     */
    adf->glyphMinX = I1616_TO_FLOAT(adfGlyphMinX);
    adf->glyphMinY = I1616_TO_FLOAT(adfGlyphMinY);
    adf->glyphMaxX = I1616_TO_FLOAT(adfGlyphMaxX);
    adf->glyphMaxY = I1616_TO_FLOAT(adfGlyphMaxY);


    /**
     *----Return the implicit ADF
     */
    return(adf);
}


/**
 *-----------------------------------------------------------------------------------
 *    IMPLICIT ADF RENDERING
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    DATA STRUCTURES AND UTILITIES FOR IMPLICIT ADF RENDERING
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    INTERNAL PATH REPRESENTATION
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Pen commands stored in an implicit ADF are converted into an internal path
 *    representation InternPath for efficient processing when rendering implicit ADFs.
 *    An InternPath includes the following:
 *    
 *    pathType: ADF_OUTLINE_PATH or ADF_UNIFORM_STROKE_PATH. This element indicates
 *    whether the InternPath represents an outline-based glyph (e.g., from a TrueType
 *    font) comprising a set of closed contours (ADF_OUTLINE_PATH) or a uniform-width
 *    stroke-based glyph comprising a stroke width and a set of stroke skeletons
 *    representing the centerlines of the glyph (ADF_UNIFORM_STROKE_PATH). Note that
 *    uniform-width stroke-based glyphs are rendered with round endcaps.
 *
 *    pathWidth: The stroke width of a uniform-width stroke-based glyph in ADF_I1616
 *    fixed point image coordinates. pathWidth is used only when pathType is
 *    ADF_UNIFORM_STROKE_PATH.
 *
 *    numPenCmds: The number of pen commands (i.e., ADFPenCmdFx data structures) in the
 *    internal path representation.
 *    
 *    penCmds: The array of ADFPenCmdFxs in the internal path. In the InternPath, pen
 *    commands are represented in ADF_I1616 fixed point image coordinates.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
typedef struct {
    ADF_U32      pathType;   /* ADF_OUTLINE_PATH or ADF_UNIFORM_STROKE_PATH */
    ADF_I1616    pathWidth;  /* The stroke width of an ADF_UNIFORM_STROKE_PATH */
    ADF_U32      numPenCmds; /* The number of pen commands in the internal path */
    ADFPenCmdFx *penCmds;    /* Pointer to the array of pen commands */
}    InternPath;


/**
 *-----------------------------------------------------------------------------------
 *    Create an internal path from the pen commands stored in the given implicit ADF,
 *    i.e., the ADFGlyph. Pen positions are transformed from ADF coordinates to
 *    ADF_I1616 fixed point image coordinates using the rendering transformation matrix
 *    stored in the given ADFRenderGlyphData. A pointer to the internal path is
 *    returned if the internal path is created successfully; a NULL is returned if the
 *    request cannot be satisfied.
 *
 *    CreateInternPath() operates as follows: (1) a temporary copy T of the pen
 *    commands comprising the given implicit ADF is allocated, (2) the pen commands of
 *    T are transformed from ADF coordinates to ADF_I1616 fixed point image
 *    coordinates, and (3) curvto commands in T are replaced by
 *    a sequence of lineto commands that closely approximate the corresponding curve
 *    segment (see below). See ADFAlgnZonesMAZ.h and ADFAlgnZonesMAZOutlines.c for
 *    details on the MAZ alignment zone detection and grid fitting system for
 *    outline-based glyphs; similarly, see ADFAlgnZonesMAZ.h and
 *    ADFAlgnZonesMAZStrokes.c for details on the MAZ alignment zone detection and grid
 *    fitting system for uniform-width stroke-based glyphs.
 *
 *    Curvto commands in T are not added to the internal path, but instead are replaced
 *    by a sequence of lineto commands that closely approximate the corresponding curve
 *    segment. Subdividing curve segments into line segments has significant advantages
 *    over processing curve segments directly. In particular, the resulting code (1)
 *    runs substantially faster, (2) facilitates hardware implementations, because line
 *    cells are simpler and easier to parallelize than curve cells and their processing
 *    requires fewer gates, (3) is easier to map to architectures with fixed point
 *    arithmetic and/or limited support for the complicated arithmetic (e.g., cos() and
 *    acos()) required for processing curve cells, and (4) is shorter and simpler, and
 *    therefore easier to understand and maintain.
 *
 *    Curve segments are subdivided into line segments using the approach described
 *    below.
 *
 *    Definitions
 *
 *      - Let P, Q, and C be the first endpoint, second endpoint, and control point of
 *        a quadratic Bezier curve segment in ADF_I1616 fixed point image coordinates.
 *        This curve segment can be defined parametrically by
 *
 *            F(t) = (x(t),y(t)) = (P * (1 - t^2)) + (2 * C * t * (1 - t)) + (Q * t^2)
 *
 *        Note that the first endpoint P corresponds to F(0) (i.e., the point on the
 *        curve segment where t = 0), and the second endpoint Q corresponds to F(1)
 *        (i.e., the point on the curve segment where t = 1).
 *      - Define H = (P + Q) / 2. Note that H is the midpoint between points P and Q.
 *      - Define M = P/4 + C/2 + Q/4. Note that M = F(0.5) (i.e., the point on the
 *        curve segment corresponding to t = 0.5).
 *
 *    Subdivision algorithm
 *
 *      - Let D be the Euclidean distance between M and H, i.e., D^2 = (H - M)^2. The
 *        distance D is a measure of how closely a single line segment from P to Q
 *        approximates the curve segment F(t). That is, a small value of D implies that
 *        line segment PQ provides a good approximation to curve segment F(t). Note
 *        that in the special case that P, C, and Q are colinear (i.e., F(t) is a
 *        degenerate curve segment), the value of D is 0, as expected.
 *      - The idea behind the algorithm is to treat D as an error term. That is, the
 *        distance between a curve segment and its approximating line segment must not
 *        exceed a specified threshold. If D exceeds the threshold, a curve segment can
 *        be subdivided into more line segments to reduce the error. The goal is to
 *        find the smallest number of line segments such that the maximum distance
 *        between each line segment and its corresponding curve segment (i.e., the
 *        portion of the original curve segment F(t) spanned by the line segment) does
 *        not exceed the threshold.
 *      - For simplicity, the algorithm assumes that curve segments are subdivided into
 *        line segments whose endpoints are computed uniformly in parameter space. That
 *        is, if N line segments are used to represent a curve segment F(t), then the
 *        first line segment has endpoints F(0) and F(1/N), the second line segment has
 *        endpoints F(1/N) and F(2/N), and so on.
 *      - A simple formula relates the threshold T (which is specified in ADF_I1616
 *        fixed point image coordinates) and the minimum number of line segments N
 *        required to represent a curve segment:
 *
 *            N = 1 + sqrt(D / T)
 *
 *      - In summary, given the coordinates P, C, and Q for a curve segment, it is
 *        straightforward to compute the number of line segments required so that the
 *        maximum distance between the curve segment and its approximating line
 *        segments is no greater than the threshold T.
 *
 *    Algorithm notes
 *
 *      - The above formula is derived by fitting a curve to a large dataset
 *        obtained by measuring the values of D and N for many typefaces. The
 *        details of this process are as follows. Given a typeface, a point size,
 *        and a threshold T, a software program analyzes each curve segment of every
 *        glyph in the typeface. For each curve segment, the software program
 *        computes the value of D and performs a brute force computation of N, the
 *        minimum number of line segments such that the maximum distance between the
 *        curve segment and the line segments is at most T. This analysis is
 *        performed across a large set of point sizes, typefaces, and values of T.
 *        The results are compiled into a table, a curve is fit to the data, and the
 *        obtained formula is shown above.
 *      - The number of line segments required (i.e., N) is proportional to the
 *        reciprocal square root of the threshold (i.e., T). This means, for example,
 *        that decreasing the threshold by a factor of 2 increases the number of line
 *        segments by a factor of sqrt(2). In the limit, as the threshold approaches
 *        zero, the number of line segments approaches infinity, and the image produced
 *        using approximating line segments converges to the image produced by
 *        processing curve segments directly.
 *      - Consider a density image produced by processing curve segments directly
 *        compared to a density image produced by processing their approximating line
 *        segments. The maximum difference in density values for any pixel in these two
 *        images is Delta = ((T - minDist) / (maxDist - minDist)) ^ gamma, where
 *        minDist is the CSM outside cutoff value, maxDist is the CSM inside cutoff
 *        value, and gamma is the CSM gamma value.
 *      - The definitions above assume that all quantities are computed in ADF_I1616
 *        fixed point image coordinates.
 *      - The following experimental observations have been determined regarding
 *        threshold values:
 *          - A threshold of T = 0.2 (1/5th of a pixel) results in unacceptable
 *            artifacts in some glyphs.
 *          - A threshold of T = 0.1 (1/10th of a pixel) is a good compromise between
 *            quality and performance. There is a noticeable performance improvement
 *            over processing curve segments directly without a perceivable degradation
 *            in quality.
 *          - A threshold of T = 0.025 (1/40th of a pixel) produces images that are
 *            nearly indistinguishable from images rendered using curve segments.
 *            However, the drop in performance compared to the case where T = 0.1 is
 *            not worth the gain in image quality.
 *      - The macro CURVE_TO_LINE_MAX_DIST (see below) implements the threshold T.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    CURVE DATA FOR SUBDIVIDING CURVE SEGMENTS INTO LINE SEGMENTS
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Quadratic Bezier curve segments are processed using sets of approximating line
 *    segments. The data for each curve segment is stored temporarily in a CurveData
 *    data structure, which includes the following elements:
 *
 *    nLineSegs: The number of line segments required to represent the curve segment.
 *
 *    px, py: The first endpoint of the curve segment in ADF_I1616 fixed point image
 *    coordinates.
 *
 *    cx, cy: The control point of the curve segment in ADF_I1616 fixed point image
 *    coordinates.
 *
 *    qx, qy: The second endpoint of the curve segment in ADF_I1616 fixed point image
 *    coordinates.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
typedef struct {
    ADF_U32 nLineSegs;    /* # of line segments required to represent the curve segment */
    ADF_I1616 px, py;    /* First endpoint of the curve segment (i.e., t = 0) */
    ADF_I1616 cx, cy;    /* Control point of the curve segment */
    ADF_I1616 cx2, cy2;    /* Control point of the curve segment */
    ADF_I1616 qx, qy;    /* Second endpoint of the curve segment (i.e., t = 1) */
}    CurveData;
/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    FixedMathNote: The fixed point implementation of the subdivision of quadratic
 *    Bezier curve segments into line segments differs from the floating implementation
 *    in the following ways:
 *
 *      - The threshold T is always set to 0.1 (i.e., 1/10th of a pixel) instead of
 *        using the constant CURVE_TO_LINE_MAX_DIST.
 *
 *      - Since square roots are expensive to evaluate, a lookup table (see below) is
 *        used to accelerate the subdivision algorithm.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    curvesToLinesTable[] is a lookup table that is used by CreateInternPath() to
 *    accelerate the subdivision of quadratic Bezier curve segments into line segments.
 *    The table contains 1024 entries and is precomputed as follows:
 *
 *    curvesToLinesTable[i] = 1 + floor(sqrt(5.0 * sqrt(i / 32.0))), where i is an
 *    integer that lies in the range [0, 1023]. The following discussion explains this
 *    formula.
 *
 *    Consider the quadratic Bezier curve segment defined by its first endpoint P =
 *    (px, py), its control point C = (cx, cy), and its second endpoint Q = (qx, qy).
 *    Note that P, C, and Q are 2D points in ADF_I1616 fixed point image coordinates.
 *
 *    Let N(d) be the function that computes the minimum number of line segments
 *    required to represent this curve segment when the threshold T is set to 0.1:
 *
 *        N(d) = 1 + sqrt(d / T)
 *             = 1 + sqrt(d / (0.1))
 *             = 1 + sqrt(10 * d)
 *
 *    where d is the maximum Euclidean distance between the curve segment and its
 *    approximating line segments.
 *
 *    Recall the definitions of H and M in the above description of the subdivision
 *    algorithm:
 *
 *        H = 0.5 * (P + Q)
 *        M = 0.5 * (C + 0.5 * (P + Q))
 *
 *    The maximum squared Euclidean distance dSqr between the curve segment and its
 *    approximating line segments is:
 *
 *        dSqr = d^2
 *             = (M - H)^2
 *             = (0.25 * ((2 * C) - P - Q))^2
 *             = (0.5 * (C - 0.5 * (P + Q))^2).
 *             = 0.25 * (dx^2 + dy^2)
 *             = 0.25 * Z
 *
 *    where dx = (cx - 0.5 * (px + qx)) and dy = dy = (cy - 0.5 * (py + qy)), and Z =
 *    ((dx * dx) + (dy * dy)).
 *
 *    It follows that
 *
 *        N(d) = 1 + sqrt(10 * d)
 *             = 1 + sqrt(10 * sqrt(d^2))
 *             = 1 + sqrt(10 * sqrt(dSqr))
 *             = 1 + sqrt(10 * sqrt(0.25 * Z))
 *             = 1 + sqrt(10 * 0.5 * sqrt(Z))
 *             = 1 + sqrt(5  * sqrt(Z))
 *
 *    The value of Z has been determined experimentally to be less than 32 in most
 *    cases. Therefore, the computation of N(d) can be accelerated using a lookup table
 *    that handles the cases where Z lies in the range [0, 32). This implementation
 *    uses a table size of 1024 elements.
 *
 *    Computing the table is accomplished with the following steps:
 *
 *    First, map an integer index i in the range [0, 1023] to the range [0, 31.96875]:
 *
 *        (i / 32.0)
 *
 *    Then apply the formula for N(d):
 *
 *      curvesToLinesTable[i] = 1 + floor(sqrt(5.0 * sqrt(i / 32.0)))
 *
 *    Note that fractional bits are discarded, effectively rounding the number of line
 *    segments towards zero. 
 *
 *    FixedMathNote: The computation of Z may overflow the ADF_I1616 fixed point
 *    representation. Therefore, this fixed point implementation computes and stores Z
 *    as an ADF_I2408 fixed point value. If the computed value of Z is negative, then
 *    the true value of Z overflows the ADF_I2408 fixed point representation. This very
 *    rare case is handled by setting the number of line segments to the constant
 *    MAX_LINE_SEGS.
 *
 *    The value of MAX_LINE_SEGS is determined as follows. The maximum representable
 *    ADF_I2408 value is m = 8388608.99609375. Therefore:
 *
 *        N(m) = 1 + sqrt(5 * sqrt(m))
 *             = N(m) = 1 + sqrt(5 * sqrt(8388608.99609375))
 *            ~= 121
 *
 *    Note that 121 is an exceedingly high number of line segments and that using more
 *    line segments than 121 to represent a single curve segment does not improve
 *    visual quality.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
static FS_CONST ADF_I8 curvesToLinesTable[] = {
    1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
    2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 
    3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 
    4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
    6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 
};
/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
#define MAX_LINE_SEGS 121    /* Maximum # of line segments per curve segment */
/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
static InternPath *CreateInternPath (void *libInst, ADFGlyph *adf, 
                                     ADFRenderGlyphData *renderGlyphData)
{
    ADF_U32      n;
    ADF_U32      size;
    ADF_I1616    adfToImg00;
    ADF_I1616    adfToImg01;
    ADF_I1616    adfToImg10;
    ADF_I1616    adfToImg11;
    ADF_I1616    adfToImgTx;
    ADF_I1616    adfToImgTy;
    ADF_I1616    xPrev = 0;
    ADF_I1616    yPrev = 0;
    ADF_I1616    internPathWidth=0;
    ADFPenCmdFx *ipPenCmd;
    CurveData   *curveList;
    CurveData   *curve;
    InternPath  *internPath;
    ADFPenCmdFx *tmpPenCmdList;
    ADFPenCmdFx *dstPenCmd;
    ADFPenCmdFx *ipSrcPenCmd;
    ADFPenCmd   *srcPenCmd;


    /**
     *----appInst is required for all memory allocation tasks
     */
    void *appInst = ADFGetAppInst(libInst);
    

    /**
     *----Extract the number of pen commands stored in the implicit ADF and set a
     *----pointer to the first implicit ADF pen command
     */
    ADF_U32 numOriginalPenCmds = adf->numCells;
    ADFPenCmd *adfPenCmd = (ADFPenCmd *) (((ADF_U8 *) adf) + adf->rootOffset);


    /**
     *----Initialize the number of pen commands in the internal path to the number of
     *----pen commands in the implicit ADF. This number will be augmented below to
     *----accommodate the additional lineto commands that result from subdividing
     *----curve segments.
     */
    ADF_U32 numInternalPenCmds = numOriginalPenCmds;

    
    /**
     *----Extract the necessary elements of the rendering transformation matrix to
     *----perform the transformation from ADF coordinates to ADF_I1616 fixed point
     *----image coordinates
     */
    adfToImg00 = FLOAT_TO_I1616(renderGlyphData->xform[0][0]);
    adfToImg01 = FLOAT_TO_I1616(renderGlyphData->xform[0][1]);
    adfToImgTx = FLOAT_TO_I1616(renderGlyphData->xform[0][2]);
    adfToImg10 = FLOAT_TO_I1616(renderGlyphData->xform[1][0]);
    adfToImg11 = FLOAT_TO_I1616(renderGlyphData->xform[1][1]);
    adfToImgTy = FLOAT_TO_I1616(renderGlyphData->xform[1][2]);


    /**
     *----Allocate a temporary copy of the pen commands. Return immediately if the
     *----allocation fails.
     */
#ifdef FS_MEM_DBG
    ((FS_STATE *)appInst)->memdbgid = "ADFPenCmdFxs";
#endif
    tmpPenCmdList = ADF_ALLOC(appInst, numOriginalPenCmds * sizeof(ADFPenCmdFx));
    if (!tmpPenCmdList)
    {
        ((FS_STATE *)appInst)->error = ERR_MALLOC_FAIL;
        return(0);
    }
    
    /**
     *----Transform the pen commands in the ADFPath from ADF coordinates to ADF_I1616
     *----fixed point image coordinates and store the results in tmpPenCmdList
     */
    srcPenCmd = adfPenCmd;
    dstPenCmd = tmpPenCmdList;
    for (n = 0; n < numOriginalPenCmds; ++n, ++srcPenCmd, ++dstPenCmd) {


        /**
         *----Copy the opcode and transform the (x,y) coordinates to ADF_I1616 fixed
         *----point image coordinates
         */
        ADF_I1616 x = FLOAT_TO_I1616(srcPenCmd->x);
        ADF_I1616 y = FLOAT_TO_I1616(srcPenCmd->y);
        dstPenCmd->opCode = srcPenCmd->opCode;
        dstPenCmd->x = I1616_MUL(x, adfToImg00) +
        I1616_MUL(y, adfToImg01) + adfToImgTx;
        dstPenCmd->y = I1616_MUL(x, adfToImg10) +
        I1616_MUL(y, adfToImg11) + adfToImgTy;


        /**
         *----If this pen command is a curvto command, transform the (cx,cy)
         *----coordinates to ADF_I1616 fixed point image coordinates
         */
        if (srcPenCmd->opCode == ADF_PEN_CURVTO_CMD) {
            ADF_I1616 cx = FLOAT_TO_I1616(srcPenCmd->cx);
            ADF_I1616 cy = FLOAT_TO_I1616(srcPenCmd->cy);
            dstPenCmd->cx = I1616_MUL(cx, adfToImg00) +
            I1616_MUL(cy, adfToImg01) + adfToImgTx;
            dstPenCmd->cy = I1616_MUL(cx, adfToImg10) +
            I1616_MUL(cy, adfToImg11) + adfToImgTy;
        }

        if (srcPenCmd->opCode == ADF_PEN_CUBETO_CMD) {
            ADF_I1616 cx = FLOAT_TO_I1616(srcPenCmd->cx);
            ADF_I1616 cy = FLOAT_TO_I1616(srcPenCmd->cy);
            
            ADF_I1616 cx2 = FLOAT_TO_I1616(srcPenCmd->cx2);
            ADF_I1616 cy2 = FLOAT_TO_I1616(srcPenCmd->cy2);

            dstPenCmd->cx = I1616_MUL(cx, adfToImg00) +
            I1616_MUL(cy, adfToImg01) + adfToImgTx;
            dstPenCmd->cy = I1616_MUL(cx, adfToImg10) +
            I1616_MUL(cy, adfToImg11) + adfToImgTy;
            
            dstPenCmd->cx2 = I1616_MUL(cx2, adfToImg00) +
            I1616_MUL(cy2, adfToImg01) + adfToImgTx;
            dstPenCmd->cy2 = I1616_MUL(cx2, adfToImg10) +
            I1616_MUL(cy2, adfToImg11) + adfToImgTy;

        }
    }


    /**
     *----If the implicit ADF represents a uniform-width stroke-based glyph,
     *----transform the path width from ADF coordinates to ADF_I1616 fixed point
     *----image coordinates
     */
    if (adf->pathType == ADF_UNIFORM_STROKE_PATH) {
        ADF_I1616 ADFToImageScale;


        /**
         *----In the common case, the rendering transformation matrix defines a
         *----uniform scaling in x and y with no rotation (i.e., the diagonal
         *----elements of the upper-left 2x2 entries of the matrix are equal and the
         *----remaining elements are zero). Handle this case separately to improve
         *----performance.
         */
        if ((adfToImg00 == adfToImg11) && (adfToImg01 == 0) && (adfToImg10 == 0)) {


            /**
             *----The rendering transformation matrix defines a uniform scaling in x
             *----and y with no rotation. Set ADFToImageScale to the scaling factor.
             */
            ADFToImageScale = adfToImg00;

            
        } else {


            /**
             *----The rendering transformation matrix defines a non-uniform scaling
             *----in x and y and/or a non-zero rotation angle. Set ADFToImageScale to
             *----the geometric mean of the scales in x and y, where the geometric
             *----mean of two values A and B is sqrt(A * B). FixedMathNote: The
             *----products (adfToImg00 * adfToImg11) and (adfToImg01 * adfToImg10)
             *----can exceed the maximum representable ADF_I1616 fixed point value
             *----(i.e., 32767 + 65535/65536). Therefore, compute these products and
             *----the matrix determinant as ADF_I2408 fixed point values.
             */
            ADF_I2408 determinant = I1616_MUL_I2408(adfToImg00, adfToImg11) -
            I1616_MUL_I2408(adfToImg01, adfToImg10);
            ADFToImageScale = I2408_SQRT_I1616(determinant);
        }


        /**
         *----Transform the path width to ADF_I1616 fixed point image coordinates
         */
        internPathWidth = I1616_MUL(FLOAT_TO_I1616(adf->pathWidth), ADFToImageScale);
    }


    /**
     *----If requested, perform MAZ alignment zone detection and grid fitting on the
     *----transformed pen commands
     */
    if (renderGlyphData->gridFitType == ADF_GRID_FIT_MAZ_PIXEL) {


        /**
         *----Determine if the implicit ADF represents a uniform-width stroke-based
         *----glyph or an outline-based glyph
         */
        if (adf->pathType == ADF_UNIFORM_STROKE_PATH) {


            /**
             *----The implicit ADF represents a uniform-width stroke-based glyph. If
             *----the implementation of the MAZ alignment zone detection and grid
             *----fitting system for uniform-width stroke-based glyphs is enabled,
             *----perform MAZ alignment zone detection and grid fitting on the
             *----transformed pen commands.
             */
#if (ADF_ENABLE_MAZ_UNIFORM_WIDTH_STROKES)
            {
                /**
                 *----Determine the glyph bounding box in ADF_I1616 fixed point ADF
                 *----coordinates
                 */
                ADF_I1616 aMinX = FLOAT_TO_I1616(adf->glyphMinX);
                ADF_I1616 aMaxX = FLOAT_TO_I1616(adf->glyphMaxX);
                ADF_I1616 aMinY = FLOAT_TO_I1616(adf->glyphMinY);
                ADF_I1616 aMaxY = FLOAT_TO_I1616(adf->glyphMaxY);


                /**
                 *----Determine the glyph bounding box in ADF_I1616 fixed point image
                 *----coordinates
                 */
                ADF_I1616 ixMin = I1616_MUL(aMinX, adfToImg00) +
                I1616_MUL(aMinY, adfToImg01) + adfToImgTx;
                ADF_I1616 iyMin = I1616_MUL(aMinX, adfToImg10) +
                I1616_MUL(aMinY, adfToImg11) + adfToImgTy;
                ADF_I1616 ixMax = I1616_MUL(aMaxX, adfToImg00) +
                I1616_MUL(aMaxY, adfToImg01) + adfToImgTx;
                ADF_I1616 iyMax = I1616_MUL(aMaxX, adfToImg10) +
                I1616_MUL(aMaxY, adfToImg11) + adfToImgTy;


                /**
                 *----Perform MAZ alignment zone detection and grid fitting on the
                 *----uniform-width stroke-based glyph. Return immediately if grid
                 *----fitting fails.
                 */
                if (!ADFGridFitMAZStroke(libInst, tmpPenCmdList, numOriginalPenCmds,
                    &internPathWidth, ixMin, iyMin, ixMax, iyMax)) {
                    ADF_FREE(appInst, tmpPenCmdList);
                    return(0);
                }
            }
#endif

            
        } else {

            
            /**
             *----The implicit ADF represents an outline-based glyph. If the
             *----implementation of the MAZ alignment zone detection and grid fitting
             *----system for outline-based glyphs is enabled, perform MAZ alignment
             *----zone detection and grid fitting on the transformed pen commands.
             */
#if (ADF_ENABLE_MAZ_OUTLINES)
            {
                /**
                 *----Perform MAZ alignment zone detection and grid fitting on the
                 *----outline-based glyph. Return immediately if grid fitting fails.
                 */
                if (!ADFGridFitMAZOutline(libInst, tmpPenCmdList, numOriginalPenCmds,
                    FLOAT_TO_I1616(renderGlyphData->ppem))) {
                    ADF_FREE(appInst, tmpPenCmdList);
                    return(0);
                }
            }
#endif
        }
    }


    /**
     *----Conservatively allocate space to store temporary curve data used to
     *----subdivide curve segments. The allocation is conservative because the number
     *----of curvto commands cannot exceed the total number of pen commands. Return
     *----immediately if the allocation fails.
     */
#ifdef FS_MEM_DBG
    ((FS_STATE *)appInst)->memdbgid = "CurveData";
#endif
    curveList = (CurveData *) ADF_ALLOC(appInst, numOriginalPenCmds *
    sizeof(CurveData));
    if (curveList == 0) {
        ADF_FREE(appInst, tmpPenCmdList);
        ((FS_STATE *)appInst)->error = ERR_MALLOC_FAIL;
        return(0);
    }


    /**
     *----For each curvto pen command in tmpPenCmdList, determine how many line
     *----segments are required to approximate the curve segment (i.e., nLineSegs)
     *----and add nLineSegs and the convex hull (i.e., the two endpoints and the
     *----control point) in ADF_I1616 fixed point image coordinates to the temporary
     *----curve list
     */
    curve = curveList;
    ipSrcPenCmd = tmpPenCmdList;
    for (n = 0; n < numOriginalPenCmds; n++, ipSrcPenCmd++) {


        /**
         *----Ignore lineto and moveto pen commands
         */
        if (ipSrcPenCmd->opCode == ADF_PEN_CURVTO_CMD) {



            /**
             *----Determine the curve segment's first endpoint in ADF_I1616 fixed
             *----point image coordinates (ipx, ipy), the control point in ADF_I1616
             *----fixed point image coordinates (icx, icy), and the second endpoint
             *----in ADF_I1616 fixed point image coordinates (iqx, iqy).
             */
            ADF_I1616 ipx = xPrev;
            ADF_I1616 ipy = yPrev;
            ADF_I1616 icx = ipSrcPenCmd->cx;
            ADF_I1616 icy = ipSrcPenCmd->cy;
            ADF_I1616 iqx = ipSrcPenCmd->x;
            ADF_I1616 iqy = ipSrcPenCmd->y;
            

            /**
             *----Determine the number of line segments required to represent the
             *----curve segment using the algorithm described above. The ADF_I1616
             *----fixed point values dx and dy below correspond directly to the
             *----variables dx and dy used in the documentation for
             *----curvesToLinesTable[] (see above).
             */
            ADF_I1616 dx = icx - ((ipx + iqx) >> 1);
            ADF_I1616 dy = icy - ((ipy + iqy) >> 1);


            /**
             *----Compute the ADF_I1616 fixed point value Z = ((dx * dx) + (dy *
             *----dy)), where Z corresponds directly to the variable Z in the
             *----documentation above. The squares of dx and dy may exceed the
             *----maximum representable ADF_I1616 value (i.e., 32767 + 65535/65536).
             *----Therefore, compute the squares as ADF_I2408 fixed point values and
             *----store the sum of squares as an ADF_I2408 fixed point value, whose
             *----maximum representable value is (8388608 + 255/256).
             */
            ADF_I2408 Z = I1616_MUL_I2408(dx, dx) + I1616_MUL_I2408(dy, dy);
            ADF_U32 nLineSegs;


            /**
             *----Determine if Z is 1) negative, 2) lies in the mathematical range
             *----[0, 32), or 3) lies in the mathematical range [32, +infinity). In
             *----practice, case 2 occurs the most often and is optimized using the
             *----lookup table curvesToLinesTable[]. Cases 1 and 3 occur very rarely
             *----and are handled separately.
             */
            if (Z < 0) {


                /**
                 *----Z is negative, which means that the computation of ((dx * dx) +
                 *----(dy * dy)) (see above) overflowed the ADF_I2408 fixed point
                 *----representation. Handle this very rare case by setting nLineSegs
                 *----to MAX_LINE_SEGS.
                 */
                nLineSegs = MAX_LINE_SEGS;

                
            } else if (Z < I2408_CONST_32) {

                /*lint -e571  Warning 571: Warning -- Suspicious cast */

                /**
                 *----Z lies in the mathematical range [0, 32). Therefore, computing
                 *----the required number of line segments can be optimized using the
                 *----lookup table curvesToLinesTable[] (see above). The
                 *----curvesToLinesTable[] table contains 1024 elements. Compute an
                 *----integer index that lies in the range [0, 1023] by multiplying
                 *----the ADF_I2408 fixed point value Z by 32 and converting the
                 *----result to an integer. Multiplying Z by 32 is equivalent to
                 *----shifting Z to the left by 5 bits. Converting Z from an
                 *----ADF_I2408 fixed point value to an integer is equivalent to
                 *----shifting Z to the right by 8 bits. Therefore, the integer index
                 *----is computed by shifting Z to the right by (8 - 5) = 3 bits.
                 */
                nLineSegs = (ADF_U32) curvesToLinesTable[Z >> 3];

                /*lint +e571  Warning 571: Warning -- Suspicious cast */
                
            } else {

                
                /**
                 *----Z lies in the mathematical range [32, +infinity). Since Z is at
                 *----least 32, the lookup table curvesToLinesTable[] cannot be used
                 *----to compute the required number of line segments. Therefore,
                 *----compute the required number of line segments by applying the
                 *----formula 1 + sqrt(5 * sqrt(Z)) (see above), which is equivalent
                 *----to 1 + sqrt(5) * sqrt(sqrt(Z)) = 1 + sqrt(5) * rsqrt(rsqrt(Z)),
                 *----where rsqrt(x) denotes the reciprocal square root of x (i.e.,
                 *----rsqrt(x) = 1 / sqrt(x)). This alternate formulation is used
                 *----because computing reciprocal square roots is faster than
                 *----computing square roots with the current fixed point math
                 *----implementation (see ADFFixedMath.h and ADFFixedMath.c). Note
                 *----that the conversion from an ADF_I1616 fixed point value to a
                 *----32-bit signed integer is implemented as a right shift of 16
                 *----bits.
                 */
                nLineSegs = 1 + (I1616_MUL(I1616_CONST_SQRT5,
                (I1616_RSQ(I2408_RSQ_I1616(Z)))) >> 16);
            }
                     
            
            /**
             *----Augment the number of pen commands required to represent and store
             *----the internal path
             */
            numInternalPenCmds += (nLineSegs - 1);


            /**
             *----Store the number of line segments required to represent the curve
             *----segment and the ADF_I1616 fixed point image space coordinates of
             *----the curve segment
             */
            curve->nLineSegs = nLineSegs;
            curve->px = ipx;
            curve->py = ipy;
            curve->cx = icx;
            curve->cy = icy;
            curve->qx = iqx;
            curve->qy = iqy;


            /**
             *----Advance the pointer to the next entry in the curve list
             */
            curve++;
        }
        if (ipSrcPenCmd->opCode == ADF_PEN_CUBETO_CMD) {



            /**
             *----Determine the curve segment's first endpoint in ADF_I1616 fixed
             *----point image coordinates (ipx, ipy), the control point in ADF_I1616
             *----fixed point image coordinates (icx, icy), and the second endpoint
             *----in ADF_I1616 fixed point image coordinates (iqx, iqy).
             */
            ADF_I1616 ipx = xPrev;
            ADF_I1616 ipy = yPrev;
            ADF_I1616 icx = ipSrcPenCmd->cx;
            ADF_I1616 icy = ipSrcPenCmd->cy;
            ADF_I1616 icx2 = ipSrcPenCmd->cx2;
            ADF_I1616 icy2 = ipSrcPenCmd->cy2;
            ADF_I1616 iqx = ipSrcPenCmd->x;
            ADF_I1616 iqy = ipSrcPenCmd->y;
            

            /**
             *----Determine the number of line segments required to represent the
             *----curve segment using the algorithm described above. The ADF_I1616
             *----fixed point values dx and dy below correspond directly to the
             *----variables dx and dy used in the documentation for
             *----curvesToLinesTable[] (see above).
             */
            ADF_I1616 dx = icx - ((ipx + iqx) >> 1);
            ADF_I1616 dy = icy - ((ipy + iqy) >> 1);
            ADF_I1616 dx2 = icx2 - ((ipx + iqx) >> 1);
            ADF_I1616 dy2 = icy2 - ((ipy + iqy) >> 1);


            /**
             *----Compute the ADF_I1616 fixed point value Z = ((dx * dx) + (dy *
             *----dy)), where Z corresponds directly to the variable Z in the
             *----documentation above. The squares of dx and dy may exceed the
             *----maximum representable ADF_I1616 value (i.e., 32767 + 65535/65536).
             *----Therefore, compute the squares as ADF_I2408 fixed point values and
             *----store the sum of squares as an ADF_I2408 fixed point value, whose
             *----maximum representable value is (8388608 + 255/256).
             */
            ADF_I2408 Z = I1616_MUL_I2408(dx, dx) + I1616_MUL_I2408(dy, dy);
            ADF_I2408 Z2 = I1616_MUL_I2408(dx2, dx2) + I1616_MUL_I2408(dy2, dy2);


            ADF_U32 nLineSegs;

            
            if(Z2 > Z)
                Z = Z2;

            /**
             *----Determine if Z is 1) negative, 2) lies in the mathematical range
             *----[0, 32), or 3) lies in the mathematical range [32, +infinity). In
             *----practice, case 2 occurs the most often and is optimized using the
             *----lookup table curvesToLinesTable[]. Cases 1 and 3 occur very rarely
             *----and are handled separately.
             */
            if (Z < 0) {


                /**
                 *----Z is negative, which means that the computation of ((dx * dx) +
                 *----(dy * dy)) (see above) overflowed the ADF_I2408 fixed point
                 *----representation. Handle this very rare case by setting nLineSegs
                 *----to MAX_LINE_SEGS.
                 */
                nLineSegs = MAX_LINE_SEGS;

                
            } else if (Z < I2408_CONST_32) {


                /**
                 *----Z lies in the mathematical range [0, 32). Therefore, computing
                 *----the required number of line segments can be optimized using the
                 *----lookup table curvesToLinesTable[] (see above). The
                 *----curvesToLinesTable[] table contains 1024 elements. Compute an
                 *----integer index that lies in the range [0, 1023] by multiplying
                 *----the ADF_I2408 fixed point value Z by 32 and converting the
                 *----result to an integer. Multiplying Z by 32 is equivalent to
                 *----shifting Z to the left by 5 bits. Converting Z from an
                 *----ADF_I2408 fixed point value to an integer is equivalent to
                 *----shifting Z to the right by 8 bits. Therefore, the integer index
                 *----is computed by shifting Z to the right by (8 - 5) = 3 bits.
                 */
                /*lint -e571  Warning 571: Warning -- Suspicious cast */
                nLineSegs = (ADF_U32) curvesToLinesTable[Z >> 3];
                /*lint +e571  Warning 571: Warning -- Suspicious cast */

                
            } else {

                
                /**
                 *----Z lies in the mathematical range [32, +infinity). Since Z is at
                 *----least 32, the lookup table curvesToLinesTable[] cannot be used
                 *----to compute the required number of line segments. Therefore,
                 *----compute the required number of line segments by applying the
                 *----formula 1 + sqrt(5 * sqrt(Z)) (see above), which is equivalent
                 *----to 1 + sqrt(5) * sqrt(sqrt(Z)) = 1 + sqrt(5) * rsqrt(rsqrt(Z)),
                 *----where rsqrt(x) denotes the reciprocal square root of x (i.e.,
                 *----rsqrt(x) = 1 / sqrt(x)). This alternate formulation is used
                 *----because computing reciprocal square roots is faster than
                 *----computing square roots with the current fixed point math
                 *----implementation (see ADFFixedMath.h and ADFFixedMath.c). Note
                 *----that the conversion from an ADF_I1616 fixed point value to a
                 *----32-bit signed integer is implemented as a right shift of 16
                 *----bits.
                 */
                nLineSegs = 1 + (I1616_MUL(I1616_CONST_SQRT5,
                (I1616_RSQ(I2408_RSQ_I1616(Z)))) >> 16);
            }

                     
            
            /**
             *----Augment the number of pen commands required to represent and store
             *----the internal path
             */
            numInternalPenCmds += (nLineSegs - 1);


            /**
             *----Store the number of line segments required to represent the curve
             *----segment and the ADF_I1616 fixed point image space coordinates of
             *----the curve segment
             */
        
            curve->nLineSegs = nLineSegs;
            curve->px = ipx;
            curve->py = ipy;
            curve->cx = icx;
            curve->cy = icy;
            curve->cx2 = icx2;
            curve->cy2 = icy2;
            curve->qx = iqx;
            curve->qy = iqy;


            /**
             *----Advance the pointer to the next entry in the curve list
             */
            curve++;
        }

        
        /**
         *----Record the current pen command's ADF_I1616 fixed point image
         *----coordinates
         */
        xPrev = ipSrcPenCmd->x;
        yPrev = ipSrcPenCmd->y;
    }


    /**
     *----Allocate memory for the internal path and its pen commands. Return
     *----immediately if the allocation fails.
     */
    size = sizeof(InternPath) + numInternalPenCmds * sizeof(ADFPenCmdFx);
#ifdef FS_MEM_DBG
    ((FS_STATE *)appInst)->memdbgid = "InternPath";
#endif
    internPath = (InternPath *) ADF_ALLOC(appInst, size);
    if (internPath == 0) {
        ADF_FREE(appInst, tmpPenCmdList);
        ADF_FREE(appInst, curveList);
        ((FS_STATE *)appInst)->error = ERR_MALLOC_FAIL;
        return(0);
    }


    /**
     *----Set the internal path type
     */
    internPath->pathType = adf->pathType;
    internPath->pathWidth = internPathWidth;


    /**
     *----Set the number of pen commands in the internal path and a pointer to the
     *----first pen command of the internal path
     */
    internPath->numPenCmds = numInternalPenCmds;
    ipPenCmd = internPath->penCmds = (ADFPenCmdFx *) (internPath + 1);


    /**
     *----Set a pointer to the first entry of the curve list and a pointer to the
     *----first pen command of tmpPenCmdList
     */
    curve = curveList;
    ipSrcPenCmd = tmpPenCmdList;


    /**
     *----Copy each moveto and lineto pen command into the internal path. Subdivide
     *----each curve segment (i.e., curvto pen command) into line segments (i.e., a
     *----sequence of lineto commands) uniformly in parameter space using the number
     *----of line segments stored in its corresponding entry in the curve list.
     */
    for (n = 0; n < numOriginalPenCmds; n++, ipSrcPenCmd++) {
        if (ipSrcPenCmd->opCode == ADF_PEN_MOVETO_CMD || ipSrcPenCmd->opCode == ADF_PEN_LINETO_CMD) {

            
            /**
             *----This pen command is a moveto command or a lineto command. Copy the
             *----pen command and proceed to the next pen command.
             */
            ipPenCmd->opCode = ipSrcPenCmd->opCode;
            ipPenCmd->x = ipSrcPenCmd->x;
            ipPenCmd->y = ipSrcPenCmd->y;
            ipPenCmd++;

            
        } else if(ipSrcPenCmd->opCode == ADF_PEN_CURVTO_CMD) {


            /**
             *----This pen command is a curvto command. Subdivide the curve segment
             *----into N line segments uniformly in parameter space, where N is a
             *----number determined above and stored in the current entry in the
             *----curve list. Handle separately the cases where N = 1 or N = 2 to
             *----improve performance.
             */
            switch (curve->nLineSegs) {

                
                /**
                 *----Handle the case where N = 1 separately to improve performance
                 */
                case 1:
                {
                    /**
                     *----Replace the curve segment using 1 line segment (i.e., a
                     *----lineto pen command to t = 1)
                     */
                    ipPenCmd->opCode = ADF_PEN_LINETO_CMD;
                    ipPenCmd->x = curve->qx;
                    ipPenCmd->y = curve->qy;
                    ipPenCmd++;
                    break;
                }
                    

                /**
                 *----Handle the case where N = 2 separately to improve performance
                 */
                case 2:
                {
                    /**
                     *----Append the first lineto command using point t = 0.5
                     */
                    ipPenCmd->opCode = ADF_PEN_LINETO_CMD;
                    ipPenCmd->x = (curve->cx + ((curve->px + curve->qx) >> 1)) >> 1;
                    ipPenCmd->y = (curve->cy + ((curve->py + curve->qy) >> 1)) >> 1;
                    ipPenCmd++;

 
                    /**
                     *----Append the second lineto command using point t = 1
                     */
                    ipPenCmd->opCode = ADF_PEN_LINETO_CMD;
                    ipPenCmd->x = curve->qx;
                    ipPenCmd->y = curve->qy;
                    ipPenCmd++;
                    break;
                }

                    
                /**
                 *----Handle the general case for N > 2
                 */
                default:
                {
                    ADF_I1616 t;
                    ADF_U32 i;


                    /**
                     *----Compute the difference in the parameter t between
                     *----successive line segments. Use the lookup table rcpTable[]
                     *----(see ADFFixedMath.h) to compute the reciprocal. Note that
                     *----rcpTable[] contains 512 elements and that curve->nLineSegs
                     *----cannot be greater than MAX_LINE_SEGS, which is less than
                     *----512.
                     */
                    FS_CONST ADF_I1616 dt = rcpTable[curve->nLineSegs];


                    /**
                     *----Precompute constant coefficients needed to evaluate the
                     *----parametric equation of the quadratic Bezier curve segment:
                     *----x(t) = (xtt * t^2) + (xt * t) + xBias,
                     *----y(t) = (ytt * t^2) + (yt * t) + yBias
                     */
                    FS_CONST ADF_I1616 xtt = curve->px + curve->qx - (curve->cx << 1);
                    FS_CONST ADF_I1616 xt = ((curve->cx - curve->px) << 1);
                    FS_CONST ADF_I1616 xBias = curve->px;
                    FS_CONST ADF_I1616 ytt = curve->py + curve->qy - (curve->cy << 1);
                    FS_CONST ADF_I1616 yt = ((curve->cy - curve->py) << 1);
                    FS_CONST ADF_I1616 yBias = curve->py;


                    /**
                     *----Determine the first N - 1 line segments (i.e., lineto
                     *----commands)
                     */
                    for (i = 1, t = dt; i < curve->nLineSegs; i++) {


                        /**
                         *----Compute the point on the curve segment corresponding to
                         *----the current value of t and append a lineto pen command
                         *----to the internal path
                         */
                        ipPenCmd->opCode = ADF_PEN_LINETO_CMD;
                        ipPenCmd->x = xBias + I1616_MUL(t, (xt + I1616_MUL(t, xtt)));
                        ipPenCmd->y = yBias + I1616_MUL(t, (yt + I1616_MUL(t, ytt)));
                        

                        /**
                         *----Advance the pointer to the next internal pen command
                         */
                        ipPenCmd++;


                        /**
                         *----Incrementally update parameter t
                         */
                        t += dt;
                    }


                    /**
                     *----Append the final lineto command for this curve segment to
                     *----the internal path. The point for this lineto command
                     *----corresponds to t = 1 (i.e., the endpoint of the curve
                     *----segment itself).
                     */
                    ipPenCmd->opCode = ADF_PEN_LINETO_CMD;
                    ipPenCmd->x = curve->qx;
                    ipPenCmd->y = curve->qy;
                    ipPenCmd++;
                    break;
                }
            }


            /**
             *----Advance the pointer to the next curve in the curve list
             */
            curve++;
        }
        else if(ipSrcPenCmd->opCode == ADF_PEN_CUBETO_CMD) {


            /**
             *----This pen command is a cubeto command. Subdivide the curve segment
             *----into N line segments uniformly in parameter space, where N is a
             *----number determined above and stored in the current entry in the
             *----curve list. Handle separately the cases where N = 1 or N = 2 to
             *----improve performance.
             */
            switch (curve->nLineSegs) {

                
                /**
                 *----Handle the case where N = 1 separately to improve performance
                 */
                case 1:
                
                
                {
                    /**
                     *----Replace the curve segment using 1 line segment (i.e., a
                     *----lineto pen command to t = 1)
                     */
                    ipPenCmd->opCode = ADF_PEN_LINETO_CMD;
                    ipPenCmd->x = curve->qx;
                    ipPenCmd->y = curve->qy;
                    ipPenCmd++;
                    break;
                }
                    
       
                /**
                 *----Handle the general case for N > 2
                 */
                default:
                {
                    ADF_I1616 t;
                    ADF_U32 i;


                    /**
                     *----Compute the difference in the parameter t between
                     *----successive line segments. Use the lookup table rcpTable[]
                     *----(see ADFFixedMath.h) to compute the reciprocal. Note that
                     *----rcpTable[] contains 512 elements and that curve->nLineSegs
                     *----cannot be greater than MAX_LINE_SEGS, which is less than
                     *----512.
                     */
                    FS_CONST ADF_I1616 dt = rcpTable[curve->nLineSegs];


                    /**
                     *----Precompute constant coefficients needed to evaluate the
                     *----parametric equation of the quadratic Bezier curve segment:
                     *----x(t) = (xttt * t^3) + (xtt * t^2) + (xt * t) + xBias,
                     *----y(t) = (ytt * t^2) + (yt * t) + yBias
                     */
                    FS_CONST ADF_I1616 xttt =  -curve->px  + (curve->cx*3) - (curve->cx2*3) + curve->qx;
                    FS_CONST ADF_I1616 xtt = (3*curve->px) - (curve->cx*6) + (curve->cx2*3);
                    FS_CONST ADF_I1616 xt = -(3*curve->px) + (curve->cx*3);
                    FS_CONST ADF_I1616 xBias =  curve->px;
                    FS_CONST ADF_I1616 yttt =  -curve->py  + (curve->cy*3) - (curve->cy2*3) + curve->qy;
                    FS_CONST ADF_I1616 ytt = (3*curve->py) - (curve->cy*6) + (curve->cy2*3);
                    FS_CONST ADF_I1616 yt = -(3*curve->py) + (curve->cy*3);
                    FS_CONST ADF_I1616 yBias = curve->py;


                    /**
                     *----Determine the first N - 1 line segments (i.e., lineto
                     *----commands)
                     */
                    for (i = 1, t = dt; i < curve->nLineSegs; i++) {
                        /**
                         *----Compute the point on the curve segment corresponding to
                         *----the current value of t and append a lineto pen command
                         *----to the internal path
                         */
                        ipPenCmd->opCode = ADF_PEN_LINETO_CMD;
                        ipPenCmd->x = xBias + I1616_MUL(t, (xt + I1616_MUL(t, (xtt + I1616_MUL(t, xttt)))));
                        ipPenCmd->y = yBias + I1616_MUL(t, (yt + I1616_MUL(t, (ytt + I1616_MUL(t, yttt)))));
                        
                        /**
                         *----Advance the pointer to the next internal pen command
                         */
                        ipPenCmd++;


                        /**
                         *----Incrementally update parameter t
                         */
                        t += dt;
                    }


                    /**
                     *----Append the final lineto command for this curve segment to
                     *----the internal path. The point for this lineto command
                     *----corresponds to t = 1 (i.e., the endpoint of the curve
                     *----segment itself).
                     */
                    ipPenCmd->opCode = ADF_PEN_LINETO_CMD;
                    ipPenCmd->x = curve->qx;
                    ipPenCmd->y = curve->qy;
                    ipPenCmd++;
                    break;
                }
            }


            /**
             *----Advance the pointer to the next curve in the curve list
             */
            curve++;
        }
    }


    /**
     *----Free the temporary curve list and the temporary pen command list
     */
    ADF_FREE(appInst, curveList);
    ADF_FREE(appInst, tmpPenCmdList);


    /**
     *----Return the internal path
     */
    return(internPath);
}


/**
 *-----------------------------------------------------------------------------------
 *    Destroy the given internal path
 *-----------------------------------------------------------------------------------
 */
static ADF_Void DestroyInternPath (void *libInst, InternPath *internPath)
{
    void *appInst = ADFGetAppInst(libInst);
    ADF_FREE(appInst, internPath);
}


/**
 *-----------------------------------------------------------------------------------
 *    DISTANCE BUFFERS
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    A distance buffer is a 2D array of ADF_I1616 fixed point values used to combine
 *    sampled distances from contributing elements of an internal path during implicit
 *    ADF rendering. Each sample point in the distance buffer corresponds to a single
 *    pixel for CRT rendering or a single pixel component for LCD rendering. Sample
 *    positions are specified in ADF_I1616 fixed point image coordinates; sample values
 *    in the distance buffer are indexed with integer indices.
 *
 *    In order to accommodate various standard display modes (such as CRT mode, where
 *    samples are spaced at one pixel intervals in x and y, or vertically striped LCD
 *    modes, where samples are spaced at one pixel intervals in y and 1/3 pixel
 *    intervals in x) and more general non-uniform pixel grids where the vertical
 *    sampling rate differs from the horizontal sampling rate, the distance buffer 
 *    header specifies the pixel spacing in x and y in ADF_I1616 fixed point image 
 *    coordinates. The inverses of these values are also stored to avoid computing them 
 *    during rendering.
 *
 *    The distance buffer is stored as a contiguous array and consists of a header,
 *    i.e., DistBuffer, followed immediately by the distance samples stored in bottom
 *    to top, left to right row major order as a linear array of ADF_I1616 fixed point
 *    values. Samples of the distance buffer are addressed using a Cartesian coordinate
 *    system, where y increases from bottom to top and the origin (0, 0) is located at
 *    the bottom left corner of the distance buffer. The distance buffer header
 *    includes the following:
 *    
 *    w, h: The width and height of the distance buffer, i.e., the number of samples in
 *    in x and y, respectively, in the distance buffer.
 *
 *    sampleSpacingX, sampleSpacingY: The spacing between each sample in the x and
 *    y-directions, respectively.
 *
 *    invSampleSpacingX, invSampleSpacingY: The inverse of sampleSpacingX and 
 *    sampleSpacingY, respectively (i.e., 1/sampleSpacingX and 1/sampleSpacingY).
 *
 *    base: A pointer to the first sampled distance value in the distance buffer.
 *
 *    Note: For consistent sampling between different display modes (e.g., CRT and
 *    LCD), this implementation places the sample point corresponding to the center of
 *    a pixel with integer pixel coordinates (i, j) at the ADF_I1616 fixed point image
 *    position (x, y) = (I32_TO_I1616(i), I32_TO_I1616(j)). For CRT rendering, (x, y)
 *    is the position of the single sample point used to determine the density of the
 *    pixel (i, j). For LCD rendering with vertical RGB stripes, (x, y) is the position
 *    of the sample point used to determine the green component of the pixel (i, j),
 *    while (x - 1/3, y) and (x + 1/3, y) are the positions of the sample points used
 *    to determine the red and blue components of the pixel (i, j), respectively.
 *    Similarly, for LCD rendering with horizontal RGB stripes (where RGB striping
 *    specifies top to bottom striping), (x, y) is the position of the sample point
 *    used to determine the green component of the pixel (i, j), while (x, y + 1/3) and
 *    (x, y - 1/3) are the positions of the sample points used to determine the red and
 *    blue components of the pixel (i, j), respectively. This convention is used when
 *    packing density values from the distance buffer into the density image during
 *    implicit ADF rendering. This convention has some consequences at the boundaries
 *    of the density image when performing LCD rendering. For example, when the display
 *    mode is ADF_REND_MODE_RGBv, the R component of the leftmost column of image
 *    pixels falls outside of the distance buffer and hence it is always set to zero,
 *    while the rightmost samples of the distance buffer are computed but not used (the
 *    rightmost samples are R components for the column immediately following the
 *    rightmost column of the distance buffer).
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    FixedMathNote: LARGE_OUTSIDE_DIST_VAL is a distance value that maps to zero
 *    density. The floating point implementation defines LARGE_OUTSIDE_DIST_VAL as the
 *    floating point constant -1000.0f. This value is unsuitable for the fixed point
 *    implementation because the ProcessCornerCell() function requires computing
 *    squares of distance values, and the square of (-1000) overflows the ADF_I1616
 *    fixed point representation. Consequently, the fixed point implementation defines
 *    LARGE_OUTSIDE_DIST_VAL as the mathematical value -128, whose square does not
 *    overflow the ADF_I1616 fixed point representation.
 *
 *    The choice of -128 instead of -1000 does not affect how outline-based glyphs are
 *    processed, because the CSM inside and outside cutoff values are required to lie
 *    in the mathematical range [-20,20]. However, the choice of -128 instead of -1000
 *    does have a subtle consequence on how uniform-width stroke-based glyphs are
 *    processed. Before processing the elements of the internal path of a uniform-width
 *    stroke-based glyph, the CSM outside and inside cutoff values are both decreased
 *    by half the stroke width (see step (a) of the Overview section at the top of this
 *    file). While the user-specified CSM outside and inside cutoff values are required
 *    to lie in the range [-20,20], the adjusted CSM outside and inside cutoff values
 *    may be less than -128 if the stroke radius is greater than 108, thereby causing
 *    the implicit ADF to be processed incorrectly. To prevent this from happening, the
 *    functions ADFRenderGlyphImplicit() and ADFSetGlyphDistMapImplicit() (see below)
 *    clamp the stroke radius to a maximum value of MAX_STROKE_RADIUS (i.e., 108).
 *-----------------------------------------------------------------------------------
 */
#define LARGE_OUTSIDE_DIST_VAL  I1616_CONST_NEG_128 /* See the fixed math note above */
#define LARGE_INSIDE_DIST_VAL   I1616_CONST_128
#define MAX_STROKE_RADIUS       I1616_CONST_108     /* See the fixed math note above */
/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
typedef struct {
    ADF_U32   w;                 /* Width of the distance buffer (# samples in x)  */
    ADF_U32   h;                 /* Height of the distance buffer (# samples in y) */
    ADF_I1616 sampleSpacingX;    /* The spacing between samples in the x-direction */
    ADF_I1616 sampleSpacingY;    /* The spacing between samples in the y-direction */
    ADF_I1616 invSampleSpacingX; /* 1 / sampleSpacingX */
    ADF_I1616 invSampleSpacingY; /* 1 / sampleSpacingY */
    ADF_I1616 *base;             /* Pointer to the distance sample values */
}    DistBuffer;


/**
 *-----------------------------------------------------------------------------------
 *    Create a distance buffer corresponding to the given display mode (e.g.,
 *    ADF_REND_MODE_CRT or ADF_REND_MODE_RGBv) and initialize distances in the distance
 *    buffer to LARGE_OUTSIDE_DIST_VAL. A pointer to the distance buffer is returned if 
 *    the distance buffer is created successfully; a NULL is returned if the request 
 *    cannot be satisfied. To support more general non-uniform pixel grids (i.e., grids 
 *    that cannot be specified with a valid display mode), allocate and initialize the 
 *    DistBuffer directly.
 *-----------------------------------------------------------------------------------
 */
static DistBuffer *CreateDistBuffer (void *libInst, ADFImage *image, 
                                     ADF_U32 displayMode)
{
    ADF_U32      size;
    ADF_U32      w;
    ADF_U32      h;
    ADF_I1616    sampleSpacingX;
    ADF_I1616    sampleSpacingY;
    ADF_I1616    invSampleSpacingX;
    ADF_I1616    invSampleSpacingY;
    DistBuffer   *distBuf;

    
    /**
     *----appInst is required for all memory allocation tasks
     */
    void *appInst = ADFGetAppInst(libInst);


    /**
     *----Set parameters for the distance buffer
     */
    switch (displayMode) {


        /**
         *----A distance buffer to support CRT rendering. Determine the distance  
         *----buffer parameters for 1 sample per pixel.
         */
        case ADF_REND_MODE_CRT:
        {
            w = image->w;
            h = image->h;
            sampleSpacingX = I1616_CONST_1;
            sampleSpacingY = I1616_CONST_1;
            invSampleSpacingX = I1616_CONST_1;
            invSampleSpacingY = I1616_CONST_1;
            break;
        }


        /**
         *----A distance buffer to support LCD rendering (i.e., sub-pixel rendering) 
         *----with vertical striping. Determine the distance buffer parameters for 
         *----3 vertical samples per pixel.
         */
        case ADF_REND_MODE_RGBv:
        case ADF_REND_MODE_BGRv:
        {
            w = 3 * image->w;
            h = image->h;
            sampleSpacingX = I1616_CONST_ONE_THIRD;
            sampleSpacingY = I1616_CONST_1;
            invSampleSpacingX = I1616_CONST_3;
            invSampleSpacingY = I1616_CONST_1;
            break;
        }


        /**
         *----A distance buffer to support LCD rendering (i.e., sub-pixel rendering) 
         *----with horizontal striping. Determine the distance buffer parameters for 
         *----3 horizontal samples per pixel.
         */
        case ADF_REND_MODE_RGBh:
        case ADF_REND_MODE_BGRh:
        {
            w = image->w;
            h = 3 * image->h;
            sampleSpacingX = I1616_CONST_1;
            sampleSpacingY = I1616_CONST_ONE_THIRD;
            invSampleSpacingX = I1616_CONST_1;
            invSampleSpacingY = I1616_CONST_3;
            break;
        }


        /**
         *----An invalid display mode
         */
        default:
        {
            return(0);
        }
    }


    /**
     *----Allocate the distance buffer and set the distance buffer parameters
     */
    size = sizeof(DistBuffer) + sizeof(ADF_I1616) * w * h;
#ifdef FS_MEM_DBG
    ((FS_STATE *)appInst)->memdbgid = "DistBuffer";
#endif
    distBuf = (DistBuffer *) ADF_ALLOC(appInst, size);
    if (distBuf == 0)
    {
        ((FS_STATE *)appInst)->error = ERR_MALLOC_FAIL;
        return(0);
    }
    distBuf->w = w;
    distBuf->h = h;
    distBuf->sampleSpacingX = sampleSpacingX;
    distBuf->sampleSpacingY = sampleSpacingY;
    distBuf->invSampleSpacingX = invSampleSpacingX;
    distBuf->invSampleSpacingY = invSampleSpacingY;
    distBuf->base = (ADF_I1616 *) (distBuf + 1);




    /**
     *----Return the distance buffer
     */
    return(distBuf);
}


/**
 *-----------------------------------------------------------------------------------
 *    Destroy the given distance buffer
 *-----------------------------------------------------------------------------------
 */
static ADF_Void DestroyDistBuffer (void *libInst, DistBuffer *distBuf)
{
    void *appInst = ADFGetAppInst(libInst);
    ADF_FREE(appInst, distBuf);
}

/**
 *-----------------------------------------------------------------------------------
 *    Create a winding number buffer to hold winding number values.
 *    Allocate the winding number buffer to have the same dimensions as the
 *    associated distance buffer and initialize all entries to 0. The winding number 
 *    buffer stores the actual winding number of each sample point, i.e. the number
 *    of contours that enclose that sample point. This value is based on the incremental 
 *    changes that occur to the winding number as a scanline is traversed from left to 
 *    right. A pointer to the winding number buffer is returned if the winding number 
 *    buffer is created successfully; a NULL is returned if the request cannot be 
 *    satisfied. 
 *-----------------------------------------------------------------------------------
 */
static ADF_I8 *CreateWindingNumberBuffer (void *libInst, DistBuffer *distBuf)
{
    
    /**
     *----appInst is required for all memory allocation tasks
     */
    void *appInst = ADFGetAppInst(libInst);

    /**
    *----Allocate a winding number buffer that has the same dimensions as the
    *----distance buffer and initialize all entries to 0. 
    */
    ADF_I8 *wnb;
#ifdef FS_MEM_DBG
    ((FS_STATE *)appInst)->memdbgid = "winding_number_buffer";
#endif
    wnb = (ADF_I8 *) ADF_CALLOC(appInst, distBuf->w * distBuf->h, sizeof(ADF_I8));
    if (wnb == 0) 
    {
        ((FS_STATE *)ADFGetAppInst(libInst))->error = ERR_MALLOC_FAIL;
    }

    /**
     *----Return the winding number buffer
     */
    return(wnb);
}


/**
 *-----------------------------------------------------------------------------------
 *    Destroy the given winding number buffer
 *-----------------------------------------------------------------------------------
 */
static ADF_Void DestroyWindingNumberBuffer (void *libInst, ADF_I8 *wnb)
{
    void *appInst = ADFGetAppInst(libInst);
    ADF_FREE(appInst, wnb);
}


/**
 *-----------------------------------------------------------------------------------
 *    IMPLICIT ADF BOUNDARY CELL DATA
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    When rendering an implicit ADF, each element of its internal path is processed
 *    by first generating an implicit ADF boundary cell representing the element and 
 *    then rendering the implicit ADF boundary cell into a distance buffer. An implicit 
 *    ADF boundary cell includes a representation of the cell's geometry (e.g., its 
 *    vertices) and data required to compute the minimum unsigned Euclidean distance 
 *    from any point inside the cell to the element represented by the cell. To render 
 *    the implicit ADF boundary cell into the distance buffer, the implicit ADF 
 *    boundary cell is rasterized to determine sample points inside the cell, distances 
 *    from each sample point to the element are computed, and the computed distances 
 *    are combined with corresponding distances in the distance buffer.
 *
 *    There are two types of implicit ADF boundary cells: cells representing straight
 *    line segments and cells representing corners of the internal path. Data required
 *    to generate and render each of these cell types are stored in a CellData data
 *    structure. This data structure stores up to two positions in ADF_I1616 fixed
 *    point image coordinates (i.e., (x0, y0) and (x1, y1)) and up to one unit (i.e.,
 *    unit length) normal vector (i.e., (nx, ny)). A summary of the data stored for
 *    each cell type follows:
 *    
 *    Implicit ADF line cell data: x0, y0, x1, y1, nx, ny -- Contains data required to
 *    generate and render an implicit ADF line cell for a line segment extending from
 *    (x0, y0) to (x1, y1) with unit normal vector (nx, ny). The unit normal vector is
 *    determined by dividing the vector (y1 - y0, -x1 + x0) by its length which
 *    provides a right handed unit normal vector (i.e., the unit normal vector points
 *    to the right as you proceed from (x0, y0) to (x1, y1)).
 *
 *    Implicit ADF corner data: x0, y0 -- Contains data required to generate and render
 *    an implicit ADF corner cell for a corner point located at (x0, y0).
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
typedef struct {
    ADF_I1616 x0;    /* x-coordinate of a 1st position (see specific cell type) */
    ADF_I1616 y0;    /* y-coordinate of a 1st position (see specific cell type) */
    ADF_I1616 x1;    /* x-coordinate of a 2nd position (see specific cell type) */
    ADF_I1616 y1;    /* y-coordinate of a 2nd position (see specific cell type) */
    ADF_I1616 nx;    /* x-coordinate of a normal vector (see specific cell type) */
    ADF_I1616 ny;    /* y-coordinate of a normal vector (see specific cell type) */    
    ADF_I1616 pre_nx;          
    ADF_I1616 pre_ny;     
    ADF_I1616 next_nx;
    ADF_I1616 next_ny;                
}    CellData;


/**
 *-----------------------------------------------------------------------------------
 *    IMPLICIT ADF RENDERING IMPLEMENTATION
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Render the given implicit ADF glyph into the specified density image using the    
 *    rendering data ADFRenderGlyphData determined by ADFRenderSetup(). If the request
 *    cannot be satisfied, the specified density image is cleared (i.e., each pixel or
 *    pixel component of the density image is set to zero). See the overview at the 
 *    beginning of this file for a description of the steps used to render an implicit 
 *    ADF glyph. 
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Forward references
 *-----------------------------------------------------------------------------------
 */
static ADF_Void ClearADFImage (ADFImage *image);
static ADF_Bool SetWindingNumberBuffer (InternPath *path, DistBuffer *distBuf, 
                                        ADF_I8 *wnb);
static ADF_Void ProcessBoundaryCells_trap(InternPath *path, 
                                  ADFRenderGlyphData *renderGlyphData, 
                                  DistBuffer *distBuf, ADF_I8 *wnb, ADF_I1616 *bcell);
static ADF_Void ProcessBoundaryCells_rect(InternPath *path, 
                                  ADFRenderGlyphData *renderGlyphData, 
                                  DistBuffer *distBuf, ADF_I8 *wnb, ADF_I1616 *bcell);
static ADF_Void MapDistBufferToImage (DistBuffer *dist, 
                                  ADFRenderGlyphData *renderGlyphData, 
                                  ADF_U32 displayMode, ADFImage *image);
/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
ADF_Void ADFRenderGlyphImplicit (void *libInst, ADFGlyph *adf, 
                                 ADFRenderGlyphData *renderGlyphData, ADFImage *image)
{
    DistBuffer    *dist;
    InternPath    *internPath;
    ADF_U32        displayMode = renderGlyphData->displayMode;


    /**
     *----If the specified density image has zero area, exit immediately
     */
    if (image->w == 0 || image->h == 0) return;


    /**
     *----Clear each pixel (for CRT rendering) or each pixel component (for LCD
     *----rendering) of the specified density image to zero
     */
    ClearADFImage(image);


    /**
     *----Check for an invalid ADF
     */
    if (adf == 0) {


        /**
         *----The ADF is NULL; return
         */
        return;


    } else if (adf->version != ADF_IMPLICIT_VERSION_NUMBER) {


        /**
         *----The ADF is not an implicit ADF; return
         */
        return;
    }


    /**
     *----Create a distance buffer for combining distance values determined when
     *----rendering the implicit ADF. During creation, each sample in the distance
     *----buffer is initialized to LARGE_OUTSIDE_DIST_VAL.
     */
    dist = CreateDistBuffer(libInst, image, displayMode);
    if (dist == 0) {
        return;
    }


    /**
     *----Create an internal path from the implicit ADF (which is really a
     *----preprocessed ADFPath) by transforming pen commands (of the preprocessed
     *----ADFPath) from ADF coordinates to ADF_I1616 fixed point image coordinates.
     *----The internal path is used to generate and render implicit ADF boundary
     *----cells. If the internal path represents an outline-based glyph, the internal
     *----path is also used to rasterize the glyph interior. Curvto commands in the
     *----preprocessed ADFPath are not added to the internal path, but instead are
     *----replaced by a sequence of lineto commands that closely approximate the
     *----corresponding curve segment.
     */
    internPath = CreateInternPath(libInst, adf, renderGlyphData);
    if (internPath == 0) {
        DestroyDistBuffer(libInst, dist);
        return;
    }


    /**
     *----Render the glyph according to its path type
     */
    if (internPath->pathType == ADF_UNIFORM_STROKE_PATH) {

        ADF_U32      i;
        ADF_U32      n = dist->h * dist->w;
        ADF_I1616    *pDist = dist->base;
        
        /**
         *----This is a uniform-width stroke-based glyph. Temporarily store the CSM
         *----inside and outside filter cutoff values and then adjust the filter
         *----cutoff values in renderGlyphData to account for the stroke width.
         *----FixedMathNote: Clamp stroke radius to MAX_STROKE_RADIUS, thereby
         *----preventing ADF_I1616 fixed point overflows when processing corner cells
         *----(see the definition of MAX_STROKE_RADIUS above for more information
         *----about this issue).
         */
        ADF_I1616 strokeRadius = (internPath->pathWidth >> 1);
        ADF_F32 insideCutoff  = renderGlyphData->insideCutoff;
        ADF_F32 outsideCutoff = renderGlyphData->outsideCutoff;
        if (strokeRadius > MAX_STROKE_RADIUS) strokeRadius = MAX_STROKE_RADIUS;
        renderGlyphData->insideCutoff = 
            I1616_TO_FLOAT(FLOAT_TO_I1616(renderGlyphData->insideCutoff) - strokeRadius);
        renderGlyphData->outsideCutoff = 
            I1616_TO_FLOAT(FLOAT_TO_I1616(renderGlyphData->outsideCutoff) - strokeRadius);

        /**
        *----Initialize the distance values in the distance buffer to 
        *----LARGE_OUTSIDE_DIST_VAL
        */
        for (i = 0; i < n; i++) 
            *pDist++ = LARGE_OUTSIDE_DIST_VAL;
        
        /**
         *----Process each element of the internal path by generating an implicit ADF
         *----boundary cell for the element and then rendering the implicit ADF
         *----boundary cell into the distance buffer.
         *----Since this is a stroke path, the contour-based winding number buffer and 
         *----bcell buffers do not apply and are not passed.
         */
        ProcessBoundaryCells_rect(internPath, renderGlyphData, dist, 
                             NULL /*wnb*/, NULL /*bcell*/);

    
        /**
         *----Map distances in the distance buffer to density values, apply color
         *----reduction if enabled, and pack the final density values into pixels of
         *----the density image
         */
        MapDistBufferToImage(dist, renderGlyphData, displayMode, image);

        
        /**
         *----Reset the CSM inside and outside filter cutoff values in
         *----renderGlyphData to their original values
         */
        renderGlyphData->insideCutoff  = insideCutoff;
        renderGlyphData->outsideCutoff = outsideCutoff;

        
    } else {

        ADF_Bool bcell_needed;
        ADF_I1616 *bcell; 

        /**
        *----Create a winding number buffer to hold winding numbers absolute values.
        *----Allocate the winding number buffer to have the same dimensions as the
        *----distance buffer and initialize all entries to 0. 
        */
        ADF_I8 *wnb = CreateWindingNumberBuffer(libInst, dist);
        if (wnb == 0) {
            /**
            *----Free temporary data structures and return
            */
            DestroyDistBuffer(libInst, dist);
            DestroyInternPath(libInst, internPath);
            return;
        }


        /**
         *----This outline-based glyph consists of closed paths with a filled
         *----interior. For each sample point in the winding number buffer compute the 
         *----winding number transition value. Then convert the transition values
         *----to actual winding number values stored in the buffer.
         */
        bcell_needed = SetWindingNumberBuffer(internPath, dist, wnb);


        
        /**
        *----Create a boundary cell buffer to hold distance values for overlapping
        *----boundary cells when needed. Allocate the boundary cell buffer to have 
        *----the same dimensions as the distance buffer and initialize all entries
        *----to 0 only when there is overlapping contours. 
        */
      
        if (bcell_needed)
        {
#ifdef FS_MEM_DBG
        ((FS_STATE *)ADFGetAppInst(libInst))->memdbgid = "boundary_cell_buffer";
#endif
            bcell = (ADF_I1616 *) ADF_CALLOC(ADFGetAppInst(libInst), dist->w * dist->h, 
                                         sizeof(ADF_I1616));
            if (bcell == 0) {
                /**
                *----Free temporary data structures and return
                */
                DestroyDistBuffer(libInst, dist);
                DestroyInternPath(libInst, internPath);
                DestroyWindingNumberBuffer(libInst, wnb);
                ((FS_STATE *)ADFGetAppInst(libInst))->error = ERR_MALLOC_FAIL;
                return;
            }
        }
        else
            bcell = 0;

        /**
         *----Process each element of the internal path by generating an implicit 
         *----ADF boundary cell for the element and then rendering the implicit 
         *----ADF boundary cell into the distance buffer.
         */
        if( renderGlyphData->insideCutoff > I1616_CONST_1 ||
            renderGlyphData->outsideCutoff < I1616_CONST_NEG_1 )
            ProcessBoundaryCells_rect(internPath, renderGlyphData, dist, wnb, bcell);
        else
            ProcessBoundaryCells_trap(internPath, renderGlyphData, dist, wnb, bcell);


        /**
        *----Free the Winding Number Buffer
        */
        DestroyWindingNumberBuffer(libInst, wnb);    


        /**
        *----Free the boundary cell buffer
        */
        ADF_FREE(ADFGetAppInst(libInst), bcell);


        /**
         *----Map distances in the distance buffer to density values, apply color
         *----reduction if enabled, and pack the final density values into pixels of
         *----the density image
         */
        MapDistBufferToImage(dist, renderGlyphData, displayMode, image);
    }

    
    /**
     *----Free temporary data structures
     */
    DestroyDistBuffer(libInst, dist);
    DestroyInternPath(libInst, internPath);
}


/**
 *-----------------------------------------------------------------------------------
 *    RENDERING IMPLICIT ADF BOUNDARY CELLS
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Process each element of the specified internal path InternPath. For each line
 *    segment and corner in the path, generate an implicit ADF boundary cell, render
 *    its signed distance field, and combine the rendered distance field with
 *    distances stored in the given distance buffer.
 *    
 *    Overview
 *
 *        - ProcessBoundaryCells() operates in two passes.
 *        - In a first pass, each pen command of the internal path is processed in
 *          sequence to determine the line segments of the internal path and the data
 *          required to generated an implicit ADF boundary cell for each of these line
 *          segments (i.e., two endpoints and a unit normal vector). Each line segment
 *          is processed, i.e., its implicit ADF boundary cell is generated, rendered,
 *          and combined with distances in the distance buffer.
 *        - In a second pass, each pen command of the internal path is processed in
 *          sequence to determine the corners of the internal path and the data
 *          required to generate an implicit ADF boundary cell for each of these
 *          corners. A corner cell is processed (i.e., generated, rendered, and
 *          combined with distances in the distance buffer) at the second endpoint
 *          of each non-degenerate line segment.
 *        - For closed-loop paths, only combine distances that are outside the glyph
 *          contours, i.e. those which have a winding number of zero. This avoids
 *          problems with corners that fall within the glyph and may overlap other
 *          boundaries.
 *        - If the internal path represents a uniform-width stroke-based glyph, an
 *          additional corner cell is generated and processed at the beginning of
 *          each non-degenerate stroke skeleton, thereby rendering a round endcap at
 *          the beginning of each non-degenerate stroke skeleton. Since a corner
 *          cell is generated and processed at the second endpoint of each
 *          non-degenerate line segment, a round endcap is also rendered at the end
 *          of each non-degenerate stroke skeleton.
 *
 *    Implementation Notes
 *
 *        - The geometry of an elements implicit ADF boundary cell is constructed
 *          such that it covers the area obtained by sweeping a line segment along the
 *          section of the internal path corresponding to the element, where the swept
 *          line segment is perpendicular to the element and extends on each side of
 *          the element by the maximum of fabs(insideCutoff) and fabs(outsideCutoff)
 *          (see the implementation note below for the rationale for using the maximum
 *          of the filter cutoff values). In addition, the geometry for each cell is
 *          chosen to balance the need for fast implicit ADF boundary cell generation
 *          with the goal of minimizing the cell area to avoid computing unnecessary
 *          distances.
 *        - Although the internal path is guaranteed to lie entirely within the 
 *          boundaries of the distance buffer, implicit ADF boundary cells can extend 
 *          beyond the edges of the distance buffer (e.g., when a large filter radius 
 *          is specified). Therefore, this implementation clips implicit ADF boundary 
 *          cells that extend beyond the boundaries of the distance buffer. Note that 
 *          clipping is performed dynamically during rasterization and that no 
 *          explicit clipped shape is computed.
 *        - When rasterizing implicit ADF boundary cells, scanlines are processed
 *          bottom to top, left to right. A Cartesian coordinate system, with y
 *          increasing in the upward direction and the origin (0, 0) located at the
 *          bottom left corner, is used to address the distance buffer (and
 *          corresponding density image). ProcessBoundaryCells() and the
 *          ProcessXXXCell() functions that it invokes use the convention that sample
 *          points located exactly on left cell edges and horizontal bottom cell edges
 *          are included in the cell interior, while sample points located exactly on
 *          right cell edges and horizontal top cell edges are excluded from the cell
 *          interior. Furthermore, sample points that fall exactly on the bottommost
 *          vertices of left edges of a cell are included in the cell interior, while
 *          sample points that fall exactly on the topmost vertices of left edges of a
 *          cell are excluded from the cell interior. Following this convention avoids
 *          duplicate processing for sample points that fall exactly on vertices shared
 *          by edges of an implicit ADF boundary cell or for sample points that fall
 *          exactly on edges shared by two implicit ADF boundary cells. Note that the
 *          stated convention for excluding sample points not included by other cells
 *          (e.g., a right edge or a horizontal top edge not shared by an abutting
 *          cell) or sample points not included by other edges (e.g., a vertex of a
 *          topmost left edge of a cell) is justified because the distances at these
 *          sample points map to zero densities (see the CSM usage note in
 *          ADFTypeSystem.h).
 *        - ProcessBoundaryCells() does not process degenerate boundary cells
 *          (i.e., boundary cells that have zero area)
 *        - ProcessBoundaryCells() and the ProcessXXXCell() functions that it invokes
 *          assume that the internal path is a directed path with ink on the right when
 *          computing sampled distances. 
 *        - Distances are stored in the distance buffer and computed in each 
 *          ProcessXXXCell() function in ADF_I1616 fixed point image coordinates.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Forward references
 *-----------------------------------------------------------------------------------
 */
/* process line cell using trapezoidal line cells */
static ADF_Void ProcessLineCell_trap (CellData *cell, DistBuffer *distBuf, 
                             ADF_I1616 filterRad, ADF_I8 *wnb, ADF_I1616 *bcell, 
                             int start,ADF_I1616* p_pre_cos,ADF_I1616* p_xyt);

/* process line cell using rectangular line cells */
static ADF_Void ProcessLineCell_rect(CellData *cell, DistBuffer *distBuf, 
                             ADF_I1616 filterRad, ADF_I8 *wnb, ADF_I1616 *bcell);

/* only called for rectangular line cells */
static ADF_Void ProcessCornerCell(CellData *cell, DistBuffer *distBuf, 
                               ADF_I1616 filterRad, ADF_I8 *wnb, ADF_I8 squarecorner);

/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *----This is the version of ProcessBoundaryCells that uses 
 *----trapezoidal line cells.
 */
static ADF_Void ProcessBoundaryCells_trap (InternPath *path, ADFRenderGlyphData *renderGlyphData, 
                                  DistBuffer *distBuf, ADF_I8 *wnb, ADF_I1616 *bcell)
{
    ADF_U32 n;

    /**
     *----Set a pointer to the first pen command of the internal path
     */
    ADFPenCmdFx *penCmd = path->penCmds;

    /**
     *----Cell data used during the traversal of the internal path to store positions 
     *----and normals required for processing implicit ADF line and corner cells
     */
    CellData cell = {0};
    int start = 1; 

    ADF_I1616 last_pre_nx= 0;        
    ADF_I1616 last_pre_ny = 0;
    ADF_I1616 last_nx = 0, last_ny = 0;
    /* 0,1-x[0..1]. 2,3-y[0..1], 4 - bi_pre_tx, 5 - bi_pre_ty, 6-dxy_pre  */
    ADF_I1616 p_pre_cos = 0,p_xyt[7] = {0};

    /**
     *----Determine the maximum filter radius in ADF_I1616 fixed point image
     *----coordinates
     */
    ADF_I1616 insideCutoffAbs = 
        I1616_ABS(FLOAT_TO_I1616(renderGlyphData->insideCutoff));
    ADF_I1616 outsideCutoffAbs = 
        I1616_ABS(FLOAT_TO_I1616(renderGlyphData->outsideCutoff));
    ADF_I1616 filterRad = (insideCutoffAbs > outsideCutoffAbs) 
                          ? insideCutoffAbs : outsideCutoffAbs;

    /**
     *----Initialize cell to keep compilers happy
     */
    cell.x0 = 0;
    cell.y0 = 0;

    /**
     *----Return immediately if all boundary cells have zero area (i.e., the filter
     *----radius is zero)
     */
    if (filterRad == 0) return;
    
    /**
     *----In a first pass, process each pen command of the internal path to determine
     *----and process each line cell
     */
    for (n = 0; n < path->numPenCmds; n++, penCmd++) {

        ADF_I1616 nx;
        ADF_I1616 ny;
        ADFPenCmdFx *next_penCmd = penCmd + 1; 

        /**
         *----(x1, y1) stores the current pen position in ADF_I1616 fixed point image
         *----coordinates
         */
        cell.x1 = penCmd->x;
        cell.y1 = penCmd->y;

        /**
         *----Process each pen command according to its opCode
         */
        switch (penCmd->opCode) {

            /**
             *----Process moveto commands
             */
            case ADF_PEN_MOVETO_CMD:
            {
                /**
                 *----Set the previous pen position to the current pen position and
                 *----proceed to the next pen command
                 */
                cell.x0 = cell.x1;
                cell.y0 = cell.y1;
                start = 1;
                break;
            }

            /**
             *----Process lineto commands
             */
            case ADF_PEN_LINETO_CMD:
                {
                /**
                 *----This is a line segment; determine the line segment's unit 
                 *----normal vector
                 */                
                ADF_I1616 temp_nx;
                ADF_I1616 temp_ny;
                
                if (start == 0) /* not the first line, therefore, pre_nx, _ny, and nx, ny already there */
                {
                    /* need to check whether the line is degenarate, if it is, get another command */
                    if ((cell.x0 == cell.x1) && (cell.y0 == cell.y1))
                        break;

                    cell.pre_nx = cell.nx;
                    cell.pre_ny = cell.ny;
                    cell.nx = cell.next_nx;
                    cell.ny = cell.next_ny;
                }

                if (start == 1) /* first line of the contour */
                {
                    nx = cell.y1 - cell.y0;
                    ny = cell.x0 - cell.x1;
                
                    if (I1616_NORMALIZE(nx, ny, &(cell.nx), &(cell.ny)) == 0) 
                        break; /* degenerate line segment, so ignore it */

                    last_pre_nx = cell.nx;  /* norm of first line is the pre of last line in same contour */      
                    last_pre_ny = cell.ny;

                    if  (next_penCmd->opCode == ADF_PEN_MOVETO_CMD) /* a single line, not possible? */
                        break; 
                    else /* ADF_PEN_LINETO_CMD */
                    {
                        ADFPenCmdFx *n_penCmd = next_penCmd + 1; 
                        ADF_U32 i = n+1;
                        ADF_U16 j = 1;
                        ADF_U32 k = n+2;
                        while ( k < path->numPenCmds && n_penCmd->opCode != ADF_PEN_MOVETO_CMD )
                        {
                            i++;
                            n_penCmd++; 
                            if ( i == path->numPenCmds - 1 )
                                break;
                        }
                        n_penCmd--;     
                        temp_nx = n_penCmd->y - (n_penCmd - 1)->y;
                        temp_ny = (n_penCmd -1)->x - n_penCmd->x;         
                        
                        while ((temp_nx == 0 ) && (temp_ny == 0))
                        {
                            j++;
                            temp_nx = n_penCmd->y - (n_penCmd - j)->y;
                            temp_ny = (n_penCmd - j)->x - n_penCmd->x;       
                        }
                        I1616_NORMALIZE(temp_nx, temp_ny, &(cell.pre_nx), &(cell.pre_ny));

                        last_nx = cell.pre_nx; /* the norm of last non-degerate line */
                        last_ny = cell.pre_ny; 
                    }
                } /* if (start == 1) */
                
                if ( ( n == path->numPenCmds - 1) || /* last line of the character */
                     (next_penCmd->opCode == ADF_PEN_MOVETO_CMD) ) /* last line of the contour */
                {
                    cell.next_nx = last_pre_nx;    
                    cell.next_ny = last_pre_ny;
                }
                else
                {
                    if ( (n == path->numPenCmds - 2) || /* second last line of the character */
                         ( (next_penCmd+1)->opCode == ADF_PEN_MOVETO_CMD ) ) /* second last line of the contour */    
                    {    
                        cell.next_nx = last_nx;    
                        cell.next_ny = last_ny;    
                    }
                    else
                    {             
                        ADF_U16 i = 0;
                        temp_nx = next_penCmd->y - cell.y1;
                        temp_ny = cell.x1 - next_penCmd->x ;      
                        while  ((temp_nx == 0 ) && (temp_ny == 0))
                        { 
                            i++;
                            temp_nx = (next_penCmd+i)->y - cell.y1;
                            temp_ny = cell.x1 - (next_penCmd+i)->x ;         
                        }
                        I1616_NORMALIZE(temp_nx, temp_ny, &(cell.next_nx), &(cell.next_ny));     
                    }
                }
                /**
                 *----Process the line segment's boundary cell. Note that cell 
                 *----currently contains the line segment's first endpoint (x0, y0), 
                 *----its second endpoint (x1, y1), and its unit normal vector 
                 *----(nx, ny).
                 */
                ProcessLineCell_trap(&cell, distBuf, filterRad, wnb, bcell, start,
                        &p_pre_cos,p_xyt);
                /**
                 *----Set the previous pen position to the current pen position 
                 *----and proceed to the next pen command
                 */
                cell.x0 = cell.x1;
                cell.y0 = cell.y1;
                start = 0; 
                break;
            }
            default:  break;
        }
    }
}


/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *----This is the original Saffron version of ProcessBoundaryCells that uses 
 *----rectangular line cells.
 */
static ADF_Void ProcessBoundaryCells_rect(InternPath *path, ADFRenderGlyphData *renderGlyphData,
                                  DistBuffer *distBuf, ADF_I8 *wnb, ADF_I1616 *bcell)
{
    ADF_U32 n;
    ADF_I1616 mx=0,my=0;


    /**
     *----Set a pointer to the first pen command of the internal path
     *----Create pointers for previous and next pen commands
     */
    ADFPenCmdFx *penCmd = path->penCmds;


    /**
     *----Cell data used during the traversal of the internal path to store positions 
     *----and normals required for processing implicit ADF line and corner cells
     */
    CellData cell;


    /**
     *----If the internal path represents a uniform-width stroke-based glyph, the
     *----validSequence Boolean is used to test for valid (i.e., non-degenerate)
     *----stroke skeletons. The validSequence Boolean is unused when processing an
     *----outline-based glyph.
     */
    ADF_U32 validSequence=0;


    /**
     *----Determine the maximum filter radius in ADF_I1616 fixed point image
     *----coordinates
     */
    ADF_I1616 insideCutoffAbs = 
        I1616_ABS(FLOAT_TO_I1616(renderGlyphData->insideCutoff));
    ADF_I1616 outsideCutoffAbs = 
        I1616_ABS(FLOAT_TO_I1616(renderGlyphData->outsideCutoff));
    ADF_I1616 filterRad = (insideCutoffAbs > outsideCutoffAbs)
                          ? insideCutoffAbs : outsideCutoffAbs;


    /**
     *----Initialize cell to keep compilers happy
     */
    cell.x0 = 0;
    cell.y0 = 0;

    /**
     *----Return immediately if all boundary cells have zero area (i.e., the filter
     *----radius is zero)
     */
    if (filterRad == 0) return;


    /**
     *----In a first pass, process each pen command of the internal path to determine
     *----and process each line cell
     */
    for (n = 0; n < path->numPenCmds; n++, penCmd++) {


        /**
         *----(x1, y1) stores the current pen position in ADF_I1616 fixed point image
         *----coordinates
         */
        cell.x1 = penCmd->x;
        cell.y1 = penCmd->y;


        /**
         *----Process each pen command according to its opCode
         */
        switch (penCmd->opCode) {


            /**
             *----Process moveto commands
             */
            case ADF_PEN_MOVETO_CMD:
            {
                /**
                 *----Set the previous pen position to the current pen position and
                 *----proceed to the next pen command
                 */
                cell.x0 = cell.x1;
                cell.y0 = cell.y1;
                break;
            }


            /**
             *----Process lineto commands
             */
            case ADF_PEN_LINETO_CMD:
            {
                /**
                 *----This is a line segment; determine the line segment's unit 
                 *----normal vector
                 */
                ADF_I1616 nx = cell.y1 - cell.y0;
                ADF_I1616 ny = cell.x0 - cell.x1;
                if (I1616_NORMALIZE(nx, ny, &(cell.nx), &(cell.ny)) == 0) {


                    /**
                     *----This is a degenerate line segment, so ignore it
                     */
                    break;
                }


                /**
                 *----Process the line segment's boundary cell. Note that cell 
                 *----currently contains the line segment's first endpoint (x0, y0), 
                 *----its second endpoint (x1, y1), and its unit normal vector 
                 *----(nx, ny).
                 */
                ProcessLineCell_rect(&cell, distBuf, filterRad, wnb, bcell);


                /**
                 *----Set the previous pen position to the current pen position 
                 *----and proceed to the next pen command
                 */
                cell.x0 = cell.x1;
                cell.y0 = cell.y1;
                break;
            }
            default:  break;
        }
    }


    /**
     *----In a second pass, process each pen command of the internal path to
     *----determine and process each corner cell
     */
    penCmd = path->penCmds;
    for (n = 0; n < path->numPenCmds; n++, penCmd++) {

        ADF_I1616 x2,y2;

        /**
         *----(x1, y1) stores the current pen position in ADF_I1616 fixed point image
         *----coordinates
         */
        cell.x1 = penCmd->x;
        cell.y1 = penCmd->y;


        /**
         *----Process each pen command according to its opCode
         */
        switch (penCmd->opCode) {

            /**
             *----Process moveto commands
             */
            case ADF_PEN_MOVETO_CMD:
            {
                /**
                 *----The pen command indicates the start of a new contour or the
                 *----start of a new stroke skeleton. Set the previous pen position
                 *----to the current pen position, set the validSequence Boolean to
                 *----false, and proceed to the next pen command.
                 */
                cell.x0 = cell.x1;
                cell.y0 = cell.y1;
                validSequence = 0;
                break;
            }


            /**
             *----Process lineto commands to determine corner cells
             */
            case ADF_PEN_LINETO_CMD:
            {
                /**
                 *----Determine if the line segment is non-degenerate
                 */
                if ((cell.x0 != cell.x1) || (cell.y0 != cell.y1)) {

                    ADF_I8 square = 0;


                    /**
                     *----This is a non-degenerate line segment. If the internal path
                     *----represents a uniform-width stroke-based glyph and this line
                     *----segment is the first non-degenerate line segment in the
                     *----current stroke skeleton (i.e., validSequence is false),
                     *----then generate and process a corner cell at the first
                     *----endpoint (i.e., (x0, y0)) of this line segment.
                     *----Corner is never square.
                     */
                    if ((path->pathType == ADF_UNIFORM_STROKE_PATH) &&
                        (validSequence == 0)) 
                        ProcessCornerCell(&cell, distBuf, filterRad, NULL, 0);

                    /**
                     *----If the previous command was a moveto, then 
                     *----store the current cell position for use with
                     *----the last line command in this contour.
                     */
                    if((penCmd-1)->opCode == ADF_PEN_MOVETO_CMD)
                    {
                        mx = cell.x1;
                        my = cell.y1;
                    }

                    /**
                     *----Determine if this corner is square. 
                     *----Corner is never square for uniform stroke paths.
                     */
                    if (path->pathType != ADF_UNIFORM_STROKE_PATH)
                    {
                        if( (n == (path->numPenCmds-1)) || 
                            (penCmd+1)->opCode == ADF_PEN_MOVETO_CMD)
                        {
                            x2 = mx;
                            y2 = my;
                        }
                        else
                        {
                            x2 = (penCmd+1)->x;
                            y2 = (penCmd+1)->y;
                        }
                        square = (((cell.x1-cell.x0) + (y2 - cell.y1)) == 0) ||
                                 (((cell.y1-cell.y0) + (x2 - cell.x1)) == 0);
                    }

                    /**
                     *----Set the previous pen position to the current pen position
                     *----and set the validSequence Boolean to true
                     */
                    cell.x0 = cell.x1;
                    cell.y0 = cell.y1;
                    validSequence = 1;


                    /**
                     *----Generate and process a corner cell at the second endpoint
                     *----of this line segment and proceed to the next pen command
                     */
                    ProcessCornerCell(&cell, distBuf, filterRad, wnb, square);
                    break;


                } else {


                    /**
                     *----This is a degenerate line segment. Set the previous pen
                     *----position to the current pen position and proceed to the
                     *----next pen command.
                     */
                    cell.x0 = cell.x1;
                    cell.y0 = cell.y1;
                    break;
                }
            }
            default: break;
        }
    }
}


/**
 *-----------------------------------------------------------------------------------
 *    Process an implicit ADF line cell for a vertical line segment whose endpoints and
 *    unit normal vector are stored in the specified cell data (i.e., CellData).
 *    Rasterize the implicit ADF line cell scanline by scanline, in bottom to top, left
 *    to right order. For each sample point inside the implicit ADF line cell, compute
 *    the signed distance to the line segment and combine the distance with the 
 *    corresponding distance stored in the distance buffer by choosing the distance
 *    with the smallest magnitude. Negative outside distances may not overwrite positive
 *    inside distances and vice versa. 
 *
 *    Vertical line segments can be processed more quickly than diagonal line segments
 *    (i.e., line segments that are neither vertical nor horizontal) because (1) the
 *    implicit ADF line cell is an axis-aligned rectangle, (2) rasterizing an
 *    axis-aligned rectangle is much faster than rasterizing an oriented rectangle, and
 *    (3) computing the distance from a point to a line is much simpler if the line is
 *    vertical (e.g., no divides are needed during setup).
 *
 *    Overview
 *
 *        - Given a line segment with endpoints (xs, ys) and (xe, ye), where it is
 *          assumed that xs == xe, generate an axis-aligned rectangular implicit ADF
 *          line cell, where the center line of the rectangle is the line segment and
 *          the width of the rectangle is twice the filter radius, i.e., 2* filterRad.
 *          Determine the positions of the four vertices of the implicit ADF line cell
 *          by offsetting each endpoint horizontally by a distance equal to filterRad.
 *        - Rasterize the axis-aligned implicit ADF line cell scanline by scanline,
 *          from bottom to top, left to right. When rasterizing each scanline,
 *          incrementally update the signed distance for each interior sample point.
 *          Combine the signed distance with the corresponding distance in the distance
 *          buffer by taking the value with the smallest magnitude.
 *
 *    See the "Implementation Notes" section in the description of ProcessLineCell()
 *    for more details on the geometry of the line cell and the sign convention for the
 *    distance field.
 *
 *    This is the rectangular line cell version
 *-----------------------------------------------------------------------------------
 */
static ADF_Void ProcessVerticalLineCell_rect(CellData *cell, DistBuffer *distBuf,
                                     ADF_I1616 filterRad, ADF_I8 *wnb, ADF_I1616 *bcell)
{
    ADF_I32   i, j;
    ADF_I32   iMin, iMax;
    ADF_I32   jMin, jMax;
    ADF_I32   w = distBuf->w;
    ADF_I32   h = distBuf->h;
    ADF_I1616 yMin, yMax;
    ADF_I1616 sampleSpacingX = (distBuf->sampleSpacingX);
    /**
     *ADF_I1616 sampleSpacingY = (distBuf->sampleSpacingY);
     */
    ADF_I1616 invSampleSpacingX = (distBuf->invSampleSpacingX);
    ADF_I1616 invSampleSpacingY = (distBuf->invSampleSpacingY);
    ADF_I1616 *dist = distBuf->base, *pBcell=0;
    ADF_I1616 xs = (cell->x0);
    ADF_I1616 ys = (cell->y0);
    /**
     *ADF_I1616 x1 = (cell->x1);
     */
    ADF_I1616 ye = (cell->y1);
    ADF_I1616 d0, dIncrement;


    /**
     *----Compute the minimum and maximum x-coordinates of the implicit ADF line cell
     *----by offsetting the vertical line's x-coordinate by the filter radius
     *----horizontally in both directions. Scale the resulting ADF_I1616 fixed point
     *----image coordinates to distance buffer coordinates.
     */

    ADF_I1616 xMin;
    ADF_I1616 xMax;
    if ( (invSampleSpacingX == I1616_CONST_1))
    {
        xMin = xs - filterRad;
        xMax = xs + filterRad;
    }
    else
    {
        xMin = I1616_MUL(xs - filterRad, invSampleSpacingX);
        xMax = I1616_MUL(xs + filterRad, invSampleSpacingX);
    }
    /**
     *----Compute the minimum and maximum y-coordinates of the implicit ADF line cell
     *----and map the resulting ADF_I1616 fixed point image coordinates to distance
     *----buffer coordinates
     */
    if (ys < ye) {


        /**
         *----This line segment proceeds from bottom to top. Therefore, ys and y1 are
         *----the minimum and maximum image space y-coordinates of the line cell,
         *----respectively. For ink-on-the-right, the initial distance
         *----is negative (outside) to the left of the line and the distance increment is 
         *----positive as you move left to right.
         */
        if ( (invSampleSpacingY == I1616_CONST_1))
        {
            yMin = ys;
            yMax = ye;
        }
        else
        {
            yMin = I1616_MUL(ys, invSampleSpacingY);
            yMax = I1616_MUL(ye, invSampleSpacingY);
        }
        d0 = -1;
        dIncrement = sampleSpacingX;
    } else {

        /**
         *----This line segment is either horizontal (and therefore degenerate) or
         *----proceeds from top to bottom. Therefore, ys and ye are the maximum and
         *----minimum image space y-coordinates of the line cell, respectively.
         *----For ink-on-the-right, the initial distance is positive (inside) and the 
         *----distance increment is negative as you move left to right.
         */
        if ( (invSampleSpacingY == I1616_CONST_1))
        {
            yMin = ye;
            yMax = ys;
        }
        else
        {
            yMin = I1616_MUL(ye, invSampleSpacingY);
            yMax = I1616_MUL(ys, invSampleSpacingY);
        }
        d0 = 1;
        dIncrement = -sampleSpacingX;
    }


    /**
     *----The cell is not rasterized if it lies below the bottom of the distance
     *----buffer
     */
    if (yMax < 0) return;


    /**
     *----Determine the bottommost (i.e., the first) scanline (i.e., jMin) of the
     *----implicit ADF line cell to be rasterized (note the inclusion of bottommost
     *----horizontal edges and bottommost vertices falling exactly on a scanline)
     */
    jMin = I1616_TO_I32(yMin);
    if (yMin < 0) jMin = 0;
    else if (yMin != I32_TO_I1616(jMin)) jMin += 1;


    /**
     *----Determine the leftmost (i.e., the first) column (i.e., iMin) of the
     *----implicit ADF line cell to be rasterized. FixedMathNote: The floating point
     *----implementation includes the leftmost edges and vertices falling exactly on
     *----a column, whereas the fixed point implementation excludes the leftmost
     *----edges and vertices falling exactly on a column. Note that samples on
     *----leftmost edges and samples that fall exactly on a column lie at a distance
     *----filterRad from the line segment and hence are mapped to a zero density.
     *----Consequently, excluding the leftmost column while processing the vertical
     *----line cell does not affect the rendered image. The fixed point
     *----implementation offers two advantages over the corresponding floating point
     *----implementation: (1) a conditional statement and a type conversion are
     *----omitted, thereby accelerating setup calculations and improving performance
     *----at smaller PPEMs, and (2) the leftmost column is always omitted, thereby
     *----improving rasterization performance.
     */
    iMin = I1616_TO_I32(xMin);
    if (xMin < 0) iMin = 0;
    else iMin += 1;


    /**
     *----Determine the topmost (i.e., the last) scanline (i.e., jMax) of the 
     *----implicit ADF line cell to be rasterized (note the exclusion of topmost 
     *----horizontal edges and topmost vertices falling exactly on a scanline)
     */
    jMax = I1616_TO_I32(yMax);
    if (jMax >= h) jMax = h - 1;
    else if (yMax == I32_TO_I1616(jMax)) jMax -= 1;


    /**
     *----Determine the rightmost (i.e., the last) column (i.e., iMax) of the
     *----implicit ADF line cell to be rasterized. FixedMathNote: The floating point
     *----implementation excludes the rightmost edges and vertices falling exactly on
     *----a column, whereas the fixed point implementation includes the rightmost
     *----edges and vertices falling exactly on a column. Note that samples on
     *----rightmost edges and samples that fall exactly on a column lie at a distance
     *----filterRad from the line segment and hence are mapped to a zero density.
     *----Consequently, including the rightmost column while processing the vertical
     *----line cell does not affect the rendered image. Compared to the floating
     *----point implementation, the fixed point implementation omits a conditional
     *----statement and a type conversion, thereby accelerating setup calculations
     *----and improving performance at smaller PPEMs.
     */
    iMax = I1616_TO_I32(xMax);
    if (iMax >= w) iMax = w - 1;


    if (wnb)
    {
        /**
        *----This is a closed loop contour path for which a winding number buffer and
        *----boundary cell buffer exist. 
        */

        /**
        *----Determine the signed distance d0 from the line segment represented by the 
        *----implicit ADF line cell to the first sample point iMin of each scanline.
        *----The sign was set previously based on the line direction.
        */
        if ( (sampleSpacingX == I1616_CONST_1))
            d0 *= (xs - I32_TO_I1616(iMin));
        else
            d0 *= (xs - I1616_MUL(I32_TO_I1616(iMin), sampleSpacingX));

        /**
        *----Rasterize the implicit ADF vertical line cell from bottom to top (i.e.,
        *----jMin to jMax), left to right (i.e., iMin to iMax). Assume ink on the right
        *----For each sample point, determine the signed distance from the line segment
        *----to the sample point and combine this distance with the corresponding
        *----distance in the distance buffer according to whether the sample point is
        *----inside a contour or outside as determined by the winding number buffer.
        *----Handle overlapping interior boundary points (winding number > 1) by first
        *----storing the negative of the distance in the boundary cell buffer to indicate
        *----the first boundary distance value. Later combine that distance with a
        *----distance value from a subsequent overlapping boundary, if such exists.
        *----Treat overlapping corner boundary cells as a single boundary cell value.
        */
        for (j = jMin; j <= jMax; j++) {
            
            ADF_I1616 d, *pDist;
            ADF_I8 *pWinding;
            ADF_I32 sample = j * w + iMin;


            /**
            *----Determine the distance d from the line segment represented by the
            *----implicit ADF line cell to the first sample point iMin of the current
            *----scanline
            */
            d = d0;


            /**
            *----Set a pointer to the entry in the distance buffer corresponding to the
            *----first sample point iMin of the current scanline
            */
            pDist    = &dist[sample];
            pWinding = &wnb[sample];
            if (bcell)
                pBcell   = &bcell[sample];


            /**
            *----Process each sample in the current scanline
            */
            for (i = iMin; i <= iMax; i++) {


                /**
                *----Combine the current line distance with its corresponding distance 
                *----in the distance buffer
                */

                if (*pWinding == 1)
                {
                    if ((d >= 0) && (d < *pDist)) 
                        *pDist = d;  /* inside pixel */
                }
                else if (*pWinding == 0)
                {
                    if ((d <= 0) && (d > *pDist)) 
                        *pDist = d;  /* outside pixel */
                }
                else
                {
                    if (pBcell && d > 0)
                    {
                        /**
                        *----Overlapping boundaries.
                        */
                        if (*pBcell == 0)
                        {
                            /**
                            *----This is the first visit to this pixel and the
                            *----winding number is greater than one. We may visit
                            *----this pixel again, so store the current distance
                            *----in the boundary cell buffer in case we do, but do not set
                            *----the distance buffer because this may be an interior 
                            *----pixel with only one visit.
                            */
                            *pBcell = d;
                        }
                        else if (*pBcell > 0)
                        {
                            /**
                            *----A previous overlapping boundary cell point was set.
                            *----This is the second or higher visit to this pixel.
                            *----We have three choices for setting the distance.
                            *----These are (d,*pBcell,*pDist). We want to take
                            *----the distance in the middle and keep the smaller
                            *----value in bcell in case there are subsequent visits.
                            *----Ultimately, this process picks the larger of the
                            *----two smallest distances.
                            *----Note: on the second visit to this pixel,
                            *----*pDist==LARGE_INSIDE_DISTANCE so *pDist will always
                            *----be set to the maximum of (d,*pBcell) and *pBcell will
                            *----be set to the minimum of (d,*pBcell).
                            */
                            if(d > *pBcell)
                            {
                                if(d < *pDist)
                                    *pDist = d;
                            }
                            else
                            {
                                *pDist = *pBcell;
                                *pBcell = d;
                            }
                        }
                    } /* if (bcell) */
                }

                /**
                *----Update the pointer to the next entry in the distance buffer and
                *----update the distance value to reflect the distance between the next
                *----sample point and the line segment
                */
                pDist++;
                pWinding++;
                if (pBcell)
                    pBcell++;
                d += dIncrement;
            }
        }
    } /* if ( wnb ) */
    else
    {
        /**
        *----This is a unform stroke path for which the winding number buffer and
        *----boundary cell buffer do not exist.
        */

        /**
        *----Determine the distance d0 from the line segment represented by the implicit
        *----ADF line cell to the first sample point iMin of each scanline
        */
        if ( (sampleSpacingX == I1616_CONST_1))
           d0 = xs - I32_TO_I1616(iMin);
        else
           d0 = xs - I1616_MUL(I32_TO_I1616(iMin), sampleSpacingX);

        /**
        *----Rasterize the implicit ADF vertical line cell from bottom to top (i.e.,
        *----jMin to jMax), left to right (i.e., iMin to iMax). For each interior sample
        *----point, determine the unsigned distance from the line segment to the sample
        *----point and combine the negative of this distance with the corresponding
        *----distance in the distance buffer.
        */
        for (j = jMin; j <= jMax; j++) {


            /**
            *----Determine the distance d from the line segment represented by the
            *----implicit ADF line cell to the first sample point iMin of the current
            *----scanline
            */
            ADF_I1616 d = d0;


            /**
            *----Set a pointer to the entry in the distance buffer corresponding to the
            *----first sample point iMin of the current scanline
            */
            ADF_I1616 *pDist = &dist[j * w + iMin];


            /**
            *----Process each sample in the current scanline
            */
            for (i = iMin; i <= iMax; i++) {


                /**
                *----Determine the negative of the magnitude of the signed distance from
                *----the current sample point to the line segment and combine this
                *----negative distance with its corresponding distance in the distance
                *----buffer
                */
                ADF_I1616 dNeg = I1616_NEGABS(d);
                if (dNeg > *pDist) *pDist = dNeg;


                /**
                *----Update the pointer to the next entry in the distance buffer and
                *----update the distance value to reflect the distance between the next
                *----sample point and the line segment
                */
                pDist++;
                d -= sampleSpacingX;
            }
        }
    } /* else no wnb */
}

/**
 *-----------------------------------------------------------------------------------
 *    Process an implicit ADF line cell for a horizontal line segment whose endpoints
 *    and unit normal vector are stored in the specified cell data (i.e., CellData).
 *    Rasterize the implicit ADF line cell scanline by scanline, in bottom to top, left
 *    to right order. For each sample point inside the implicit ADF line cell, compute
 *    the signed distance to the line segment and combine the distance with the
 *    corresponding distance stored in the distance buffer by choosing the distance with
 *    the smallest magnitude.
 *
 *    As is the case with vertical line segments, horizontal line segments can be
 *    processed more quickly than diagonal line segments (i.e., line segments that are
 *    neither vertical nor horizontal) because (1) the implicit ADF line cell is an
 *    axis-aligned rectangle, (2) rasterizing an axis-aligned rectangle is much faster
 *    than rasterizing an oriented rectangle, and (3) computing the distance from a
 *    point to a line is much simpler if the line is horizontal (e.g., no divides are
 *    needed during setup).
 *
 *    Horizontal line segments can be processed even faster than vertical line
 *    segments, because in the case of horizontal line segments, the distance from a
 *    sample point to the line segment is constant for all sample points in a scanline.
 *    Therefore, the distance computation can be pulled out of the innermost loop.
 *    In principle, vertical line segments can be handled similarly by inverting the
 *    loop order (i.e., by rasterizing the line cell from left to right in the outer
 *    loop and from bottom to top in the inner loop). However, because the distance
 *    values are stored in the distance buffer in row major order, inverting the loop
 *    order decreases cache coherence.
 *
 *    Overview
 *
 *        - Given a line segment with endpoints (xs, ys) and (xe, ye), where it is
 *          assumed that y0 == y1, generate an axis-aligned rectangular implicit ADF
 *          line cell, where the center line of the rectangle is the line segment and
 *          the height of the rectangle is twice the filter radius, i.e., 2* filterRad.
 *          Determine the positions of the four vertices of the implicit ADF line cell
 *          by offsetting each endpoint vertically by a distance equal to filterRad.
 *        - Rasterize the axis-aligned implicit ADF line cell scanline by scanline,
 *          from bottom to top, left to right. For each scanline, incrementally update
 *          the signed distance from the scanline to the horizontal line segment.
 *          For each interior sample point, combine the distance with the corresponding
 *          distance in the distance buffer by taking the value with the smallest
 *          magnitude.
 *
 *    See the "Implementation Notes" section in the description of ProcessLineCell()
 *    for more details on the geometry of the line cell and the sign convention for the
 *    distance field.
 *
 *    This is the rectangular line cell version
 *-----------------------------------------------------------------------------------
 */
static ADF_Void ProcessHorizontalLineCell_rect(CellData *cell, DistBuffer *distBuf,
                                               ADF_I1616 filterRad, ADF_I8 *wnb,
                                               ADF_I1616 *bcell)
{
    ADF_I32   i, j;
    ADF_I32   iMin, iMax;
    ADF_I32   jMin, jMax;
    ADF_I32   w = distBuf->w;
    ADF_I32   h = distBuf->h;
    ADF_I1616 xMin, xMax;
    /**
     *ADF_I1616 sampleSpacingX = (distBuf->sampleSpacingX);
     */
    ADF_I1616 sampleSpacingY = (distBuf->sampleSpacingY);
    ADF_I1616 invSampleSpacingX = (distBuf->invSampleSpacingX);
    ADF_I1616 invSampleSpacingY = (distBuf->invSampleSpacingY);
    ADF_I1616 xs = (cell->x0);
    ADF_I1616 ys = (cell->y0);
    ADF_I1616 xe = (cell->x1);
    /**
     *ADF_I1616 y1 = (cell->y1);
     */
    ADF_I1616 *dist = distBuf->base, *pDist=0, *pBcell=0;
    ADF_I1616 d, dIncrement;
    ADF_I8    *pWinding;

    /**
     *----Compute the minimum and maximum y-coordinates of the implicit ADF line cell
     *----by offsetting the horizontal line's y-coordinate by the filter radius
     *----vertically in both directions. Scale the resulting ADF_I1616 fixed point
     *----image coordinates to distance buffer coordinates.
     */

    ADF_I1616 yMin;
    ADF_I1616 yMax;


    if ( (invSampleSpacingY == I1616_CONST_1))
    {
        yMin = ys - filterRad;
        yMax = ys + filterRad;
    }
    else
    {
        yMin = I1616_MUL(ys - filterRad, invSampleSpacingY);
        yMax = I1616_MUL(ys + filterRad, invSampleSpacingY);
    }

    /**
     *----Compute the minimum and maximum x-coordinates of the implicit ADF line cell
     *----and map the resulting ADF_I1616 fixed point image coordinates to distance
     *----buffer coordinates
     */
    if (xs < xe) {


        /**
         *----This line segment proceeds from left to right. Therefore, xs and xe are
         *----the minimum and maximum image space x-coordinates of the line cell,
         *----respectively. For ink-on-the-right, the initial distance
         *----is positive (inside) and the distance increment is negative as you move
         *----up in the y direction. 
         */  
        if ( (invSampleSpacingX == I1616_CONST_1))
        {    
            xMin = xs;
            xMax = xe;
        }
        else
        {
            xMin = I1616_MUL(xs, invSampleSpacingX);
            xMax = I1616_MUL(xe, invSampleSpacingX);
        }
        d = 1;
        dIncrement = -sampleSpacingY;
    } else {

        /**
         *----This line segment is either vertical (and therefore degenerate) or
         *----proceeds from right to left. Therefore, x0 and xe are the maximum and
         *----minimum image space x-coordinates of the line cell, respectively.
         *----For ink-on-the-right, the initial distance
         *----is negative (outside) and the distance increment is positive.
         */
        if ( (invSampleSpacingX == I1616_CONST_1))
        {
            xMin = xe;
            xMax = xs;
        }
        else
        {
            xMin = I1616_MUL(xe, invSampleSpacingX);
            xMax = I1616_MUL(xs, invSampleSpacingX);
        }
        d = -1;
        dIncrement = sampleSpacingY;
    }


    /**
     *----Determine the bottommost (i.e., the first) scanline (i.e., jMin) of the
     *----implicit ADF line cell to be rasterized. FixedMathNote: The floating point
     *----implementation includes the bottommost horizontal edges and bottommost
     *----vertices falling exactly on a scanline, whereas the fixed point
     *----implementation excludes the bottommost horizontal edges and bottommost
     *----vertices falling exactly on a scanline. Note that samples on bottommost
     *----horizontal edges and samples that fall exactly on a scanline lie at a
     *----distance filterRad from the line segment and hence are mapped to a zero
     *----density. Consequently, excluding the bottommost scanline while processing
     *----the horizontal line cell does not affect the rendered image. The fixed
     *----point implementation offers two advantages over the corresponding floating
     *----point implementation: (1) a conditional statement and a type conversion are
     *----omitted, thereby accelerating setup calculations and improving performance
     *----at smaller PPEMs, and (2) the bottommost scanline is always omitted,
     *----thereby improving rasterization performance.
     */
    jMin = I1616_TO_I32(yMin);
    if (yMin < 0) jMin = 0;
    else jMin += 1;


    /**
     *----Determine the leftmost (i.e., the first) column (i.e., iMin) of the
     *----implicit ADF line cell to be rasterized (note the inclusion of leftmost
     *----edges and vertices falling exactly on a column)
     */
    iMin = I1616_TO_I32(xMin);
    if (xMin < 0) iMin = 0;
    else if (xMin != I32_TO_I1616(iMin)) iMin += 1;


    /**
     *----Determine the topmost (i.e., the last) scanline (i.e., jMax) of the
     *----implicit ADF line cell to be rasterized. FixedMathNote: The floating point
     *----implementation excludes the topmost horizontal edges and topmost vertices
     *----falling exactly on a scanline, whereas the fixed point implementation
     *----includes the topmost horizontal edges and topmost vertices falling exactly
     *----on a scanline. Note that samples on topmost horizontal edges and samples
     *----that fall exactly on a scanline lie at a distance filterRad from the line
     *----segment and hence are mapped to a zero density. Consequently, including the
     *----topmost scanline while processing the horizontal line cell does not affect
     *----the rendered image. Compared to the floating point implementation, the
     *----fixed point implementation omits a conditional statement and a type
     *----conversion, thereby accelerating setup calculations and improving
     *----performance at smaller PPEMs.
     */
    jMax = I1616_TO_I32(yMax);
    if (jMax >= h) jMax = h - 1;


    /**
     *----Determine the rightmost (i.e., the last) column (i.e., iMax) of the 
     *----implicit ADF line cell to be rasterized (note the exclusion of topmost 
     *----edges and vertices falling exactly on a column)
     */
    iMax = I1616_TO_I32(xMax);
    if (iMax >= w) iMax = w - 1;
    else if (xMax == I32_TO_I1616(iMax)) iMax -= 1;


    if (wnb)
    {
        /**
        *----This is a closed loop contour path for which a winding number buffer and
        *----boundary cell buffer exist. 
        */

        /**
        *----Determine the distance d from the line segment represented by the implicit
        *----ADF line cell to the bottommost (i.e., jMin) scanline to be rasterized
        */
        if ( (sampleSpacingY == I1616_CONST_1))
            d *= (ys - I32_TO_I1616(jMin));
        else
            d *= (ys - I1616_MUL(I32_TO_I1616(jMin), sampleSpacingY));

        /**
        *----Rasterize the implicit ADF horizontal line cell from bottom to top (i.e.,
        *----jMin to jMax), left to right (i.e., iMin to iMax). Assume ink on the right.
        *----For each scanline,determine the signed distance from the line segment to 
        *----the scanline and combine this distance with the corresponding distance in
        *----the distance buffer according to whether the sample point is 
        *----inside a contour or outside as determined by the winding number buffer.
        *----Since the line segment is horizontal, the distance is
        *----constant for all samples in a scanline.
        *----Handle overlapping interior boundary points (winding number > 1) by first  
        *----storing the negative of the distance in the boundary cell buffer to indicate
        *----the first boundary distance value. Later combine that distance with a 
        *----distance value from a subsequent overlapping boundary, if such exists.
        *----Treat overlapping corner boundary cells as a single boundary cell value.
        */
        for (j = jMin; j <= jMax; j++) {

            ADF_I32 sample = j * w + iMin;

            /**
            *----Set a pointer to the entry in the distance buffer corresponding to the
            *----first sample point iMin of the current scanline
            */
            pDist    = &dist[sample];
            pWinding = &wnb[sample];
            if (bcell)
                pBcell   = &bcell[sample];


            /**
            *----Process each sample in the current scanline
            */
            for (i = iMin; i <= iMax; i++) {


                /**
                *----Combine the current line distance with its corresponding distance 
                *----in the distance buffer
                */
                if (*pWinding == 1)
                {
                    if ((d > 0) && (d < *pDist))
                        *pDist = d;  /* inside pixel */
                }
                else if (*pWinding == 0)
                {
                    if ((d <= 0) && (d > *pDist)) 
                        *pDist = d;  /* outside pixel */
                }
                else
                {
                    if (pBcell && d > 0)
                    {
                        /**
                        *----Overlapping boundaries. 
                        */
                        if (*pBcell == 0)
                        {
                            /**
                            *----This is the first visit to this pixel and the
                            *----winding number is greater than one. We may visit
                            *----this pixel again, so store the current distance
                            *----in the boundary cell buffer in case we do, but do not set
                            *----the distance buffer because this may be an interior
                            *----pixel with only one visit.
                            */
                            *pBcell = d;
                        }
                        else if (*pBcell > 0)
                        {
                            /**
                            *----A previous overlapping boundary cell point was set.
                            *----This is the second or higher visit to this pixel.
                            *----We have three choices for setting the distance.
                            *----These are (d,*pBcell,*pDist). We want to take
                            *----the distance in the middle and keep the smaller
                            *----value in bcell in case there are subsequent visits.
                            *----Ultimately, this process picks the larger of the 
                            *----two smallest distances.
                            *----Note: on the second visit to this pixel,
                            *----*pDist==LARGE_INSIDE_DISTANCE so *pDist will always
                            *----be set to the maximum of (d,*pBcell) and *pBcell will
                            *----be set to the minimum of (d,*pBcell).
                            */
                            if(d > *pBcell)
                            {
                                if(d < *pDist)
                                    *pDist = d;
                            }
                            else
                            {
                                *pDist = *pBcell;
                                *pBcell = d;
                            }
                        }
                    } /* if (bcell) */

                }

                /**
                *----Update the pointers to the next entry in the buffers
                */
                pDist++;
                pWinding++;
                if (pBcell)
                    pBcell++;

            }

            /**
            *----Update the distance value to reflect the distance between the line
            *----segment and the next scanline
            */
            d += dIncrement;
        }
    } /* if ( wnb ) */

    else

    {

        /**
        *----This is a uniform stroke path for which no winding number buffer or
        *----boundary cell buffer exists.
        */

        /**
        *----Determine the distance d from the line segment represented by the implicit
        *----ADF line cell to the bottommost (i.e., jMin) scanline to be rasterized
        */
        if ( (sampleSpacingY == I1616_CONST_1))
            d = ys - I32_TO_I1616(jMin);
        else
            d = (ys - I1616_MUL(I32_TO_I1616(jMin), sampleSpacingY));

        /**
        *----Rasterize the implicit ADF horizontal line cell from bottom to top (i.e.,
        *----jMin to jMax), left to right (i.e., iMin to iMax). For each scanline,
        *----determine the unsigned distance from the line segment to the scanline and
        *----combine the negative of this distance with the corresponding distance in
        *----the distance buffer. Since the line segment is horizontal, the distance is
        *----constant for all samples in a scanline.
        */

        for (j = jMin; j <= jMax; j++) {


            /**
            *----Determine the negative of the magnitude of the signed distance from the
            *----current scanline to the line segment. This value is constant for all
            *----samples in the scanline.
            */
            FS_CONST ADF_I1616 dNeg = I1616_NEGABS(d);


            /**
            *----Set a pointer to the entry in the distance buffer corresponding to the
            *----first sample point iMin of the current scanline
            */
            pDist = &dist[j * w + iMin];


            /**
            *----Process each sample in the current scanline
            */
            for (i = iMin; i <= iMax; i++) {


                /**
                *----Combine the negative distance with its corresponding distance in
                *----the distance buffer
                */
                if (dNeg > *pDist) *pDist = dNeg;


                /**
                *----Update the pointer to the next entry in the distance buffer
                */
                pDist++;
            }


            /**
            *----Update the distance value to reflect the distance between the line
            *----segment and the next scanline
            */
            d -= sampleSpacingY;
        }
    } /* else no wnb */
}


/**
 *-----------------------------------------------------------------------------------
 *    Process an implicit ADF line cell for a horizontal line segment whose endpoints
 *    and unit normal vector are stored in the specified cell data (i.e., CellData).
 *    Rasterize the implicit ADF line cell scanline by scanline, in bottom to top, left
 *    to right order. For each sample point inside the implicit ADF line cell, compute
 *    the signed distance to the line segment and combine the distance with the
 *    corresponding distance stored in the distance buffer by choosing the distance with
 *    the smallest magnitude.
 *
 *    As is the case with vertical line segments, horizontal line segments can be
 *    processed more quickly than diagonal line segments (i.e., line segments that are
 *    neither vertical nor horizontal) because (1) the implicit ADF line cell is an
 *    axis-aligned rectangle, (2) rasterizing an axis-aligned rectangle is much faster
 *    than rasterizing an oriented rectangle, and (3) computing the distance from a
 *    point to a line is much simpler if the line is horizontal (e.g., no divides are
 *    needed during setup).
 *
 *    Horizontal line segments can be processed even faster than vertical line
 *    segments, because in the case of horizontal line segments, the distance from a
 *    sample point to the line segment is constant for all sample points in a scanline.
 *    Therefore, the distance computation can be pulled out of the innermost loop.
 *    In principle, vertical line segments can be handled similarly by inverting the
 *    loop order (i.e., by rasterizing the line cell from left to right in the outer
 *    loop and from bottom to top in the inner loop). However, because the distance
 *    values are stored in the distance buffer in row major order, inverting the loop
 *    order decreases cache coherence.
 *
 *    Overview
 *
 *        - Given a line segment with endpoints (xs, ys) and (xe, ye), where it is
 *          assumed that y0 == y1, generate an axis-aligned rectangular implicit ADF
 *          line cell, where the center line of the rectangle is the line segment and
 *          the height of the rectangle is twice the filter radius, i.e., 2* filterRad.
 *          Determine the positions of the four vertices of the implicit ADF line cell
 *          by offsetting each endpoint vertically by a distance equal to filterRad.
 *        - Rasterize the axis-aligned implicit ADF line cell scanline by scanline,
 *          from bottom to top, left to right. For each scanline, incrementally update
 *          the signed distance from the scanline to the horizontal line segment.
 *          For each interior sample point, combine the distance with the corresponding
 *          distance in the distance buffer by taking the value with the smallest 
 *          magnitude.
 *
 *    See the "Implementation Notes" section in the description of ProcessLineCell()
 *    for more details on the geometry of the line cell and the sign convention for the
 *    distance field.
 *
 *    This is the trapezoidal line cell version
 *-----------------------------------------------------------------------------------
 */


static ADF_Void ProcessHorizontalLineCell_trap(CellData *cell, DistBuffer *distBuf,
                                       ADF_I1616 filterRad, ADF_I8 *wnb, ADF_I1616 *bcell,
                                       ADF_I1616 bi_pre_txy, ADF_I1616 bi_next_txy,
                                       ADF_I1616* x,
                                       ADF_I1616* y)

{
    ADF_I32   i, j;
    ADF_I32   iMin, iMax;
    ADF_I32   jMin, jMax;
    ADF_I32   w = distBuf->w;
    ADF_I32   h = distBuf->h;
    ADF_I1616 xMin, xMax;
    /**
     *ADF_I1616 sampleSpacingX = (distBuf->sampleSpacingX);
     */
    ADF_I1616 sampleSpacingY = (distBuf->sampleSpacingY);
    ADF_I1616 invSampleSpacingY = (distBuf->invSampleSpacingY);
    ADF_I1616 invSampleSpacingX = (distBuf->invSampleSpacingX);
    ADF_I1616 xs = (cell->x0);
    ADF_I1616 ys = (cell->y0);
    ADF_I1616 xe = (cell->x1);
    ADF_I1616 *dist = distBuf->base, *pDist=0, *pBcell=0;
    ADF_I1616 d, dIncrement;
    ADF_I8    *pWinding;
    ADF_I1616 yMin;
    ADF_I1616 yMax;
    ADF_I1616 ddx_min = 0, ddx_max = 0;
    ADF_I1616 dy;

    /**
     *----Compute the minimum and maximum y-coordinates of the implicit ADF line cell
     *----by offsetting the horizontal line's y-coordinate by the filter radius
     *----vertically in both directions. Scale the resulting ADF_I1616 fixed point
     *----image coordinates to distance buffer coordinates.
     */
    if ( (invSampleSpacingY == I1616_CONST_1))
    {
        yMin = ys - filterRad;
        yMax = ys + filterRad;
    }
    else
    {
        yMin = I1616_MUL(ys - filterRad, invSampleSpacingY);
        yMax = I1616_MUL(ys + filterRad, invSampleSpacingY);
    }

    /**
     *----Compute the minimum and maximum x-coordinates of the implicit ADF line cell
     *----and map the resulting ADF_I1616 fixed point image coordinates to distance
     *----buffer coordinates
     */
    if (xs < xe) {
        if(filterRad > I1616_CONST_1)
        {
            if (y[0] != y[3])
                y[0] = y[3] = yMin;
        }

        /**
         *----This line segment proceeds from left to right. Therefore, xs and xe are
         *----the minimum and maximum image space x-coordinates of the line cell,
         *----respectively. For ink-on-the-right, the initial distance
         *----is positive (inside) and the distance increment is negative as you move
         *----up in the y direction.
         */
        {
            if(invSampleSpacingX == I1616_CONST_1)
            {
                ddx_min = bi_pre_txy;
                ddx_max = bi_next_txy;
            }
            else
            {
                ddx_min = I1616_MUL(bi_pre_txy, invSampleSpacingX);
                ddx_max = I1616_MUL(bi_next_txy,invSampleSpacingX);
            }

            if(sampleSpacingY  != I1616_CONST_1)
            {
                ddx_min = I1616_MUL(ddx_min, sampleSpacingY);
                ddx_max = I1616_MUL(ddx_max, sampleSpacingY);
            }

            if (y[0] <= 0)
                dy = I1616_NEGATE(y[0]);
            else
            {
                dy = I32_TO_I1616(I1616_TO_I32(y[0])) - y[0];

                if (dy != 0)
                    dy = dy + I1616_CONST_1;
            }
            xMin = x[0] + I1616_MUL(dy, ddx_min);
            if(xMin < x[0] && xMin < x[1])
                return;
            if (y[3] <= 0)
                dy = I1616_NEGATE(y[3]);
            else 
            {
                dy = I32_TO_I1616(I1616_TO_I32(y[3])) - y[3];
                if (dy != 0)
                    dy = dy + I1616_CONST_1;
            }
            xMax = x[3] + I1616_MUL(dy, ddx_max);
            if(xMax > x[3] && xMax > x[2]) 
                return;
        }

        d = 1;
        dIncrement = -sampleSpacingY;
    } else {
        if(filterRad > I1616_CONST_1)
        {
            if (y[1] != y[2])
                y[1] = y[2] = yMin;
        }

        /**
         *----This line segment is either vertical (and therefore degenerate) or
         *----proceeds from right to left. Therefore, xs and xe are the maximum and
         *----minimum image space x-coordinates of the line cell, respectively.
         *----For ink-on-the-right, the initial distance
         *----is negative (outside) and the distance increment is positive.
         */
        {
            if(invSampleSpacingX == I1616_CONST_1)
            {
                ddx_min = bi_next_txy;
                ddx_max = bi_pre_txy;
            }
            else
            {
                ddx_min = I1616_MUL(bi_next_txy,invSampleSpacingX);
                ddx_max = I1616_MUL(bi_pre_txy, invSampleSpacingX);
            }

            if(sampleSpacingY  != I1616_CONST_1)
            {
                ddx_min = I1616_MUL(ddx_min, sampleSpacingY );
                ddx_max = I1616_MUL(ddx_max,sampleSpacingY );
            }

            if (y[2] <= 0)
                dy = I1616_NEGATE(y[2]);
            else
            {
                dy = I32_TO_I1616(I1616_TO_I32(y[2])) - y[2];
                if (dy != 0)
                    dy = dy + I1616_CONST_1;
            }
            xMin = x[2] + I1616_MUL(dy, ddx_min);
            if(xMin < x[2] && xMin < x[3])
                return;

            if (y[1] <= 0)
                dy = I1616_NEGATE(y[1]);
            else
            {
                dy = I32_TO_I1616(I1616_TO_I32(y[1])) - y[1];
                if (dy != 0)
                    dy = dy + I1616_CONST_1;
            }
            xMax = x[1] + I1616_MUL(dy, ddx_max);
            if(xMax > x[1] && xMax > x[0])
                return;
        }

        d = -1;
        dIncrement = sampleSpacingY;
    }

    /**
     *----Determine the bottommost (i.e., the first) scanline (i.e., jMin) of the
     *----implicit ADF line cell to be rasterized. FixedMathNote: The floating point
     *----implementation includes the bottommost horizontal edges and bottommost
     *----vertices falling exactly on a scanline, whereas the fixed point
     *----implementation excludes the bottommost horizontal edges and bottommost
     *----vertices falling exactly on a scanline. Note that samples on bottommost
     *----horizontal edges and samples that fall exactly on a scanline lie at a
     *----distance filterRad from the line segment and hence are mapped to a zero
     *----density. Consequently, excluding the bottommost scanline while processing
     *----the horizontal line cell does not affect the rendered image. The fixed
     *----point implementation offers two advantages over the corresponding floating
     *----point implementation: (1) a conditional statement and a type conversion are
     *----omitted, thereby accelerating setup calculations and improving performance
     *----at smaller PPEMs, and (2) the bottommost scanline is always omitted,
     *----thereby improving rasterization performance.
     */
    jMin = I1616_TO_I32(yMin);
    if (yMin < 0) jMin = 0;
    else if (yMin != I32_TO_I1616(jMin)) jMin += 1;

    /**
     *----Determine the leftmost (i.e., the first) column (i.e., iMin) of the
     *----implicit ADF line cell to be rasterized (note the inclusion of leftmost
     *----edges and vertices falling exactly on a column)
     */
    iMin = I1616_TO_I32(xMin);
    if (xMin < 0) iMin = 0;
    else if (xMin != I32_TO_I1616(iMin))
        iMin += 1;

    /**
     *----Determine the topmost (i.e., the last) scanline (i.e., jMax) of the
     *----implicit ADF line cell to be rasterized. FixedMathNote: The floating point
     *----implementation excludes the topmost horizontal edges and topmost vertices
     *----falling exactly on a scanline, whereas the fixed point implementation
     *----includes the topmost horizontal edges and topmost vertices falling exactly
     *----on a scanline. Note that samples on topmost horizontal edges and samples
     *----that fall exactly on a scanline lie at a distance filterRad from the line
     *----segment and hence are mapped to a zero density. Consequently, including the
     *----topmost scanline while processing the horizontal line cell does not affect
     *----the rendered image. Compared to the floating point implementation, the
     *----fixed point implementation omits a conditional statement and a type
     *----conversion, thereby accelerating setup calculations and improving
     *----performance at smaller PPEMs.
     */

    jMax = I1616_TO_I32(yMax);
    if (jMax >= h) 
        jMax = h - 1;
    else if (yMax == I32_TO_I1616(jMax)) jMax -= 1;

    /**
     *----Determine the rightmost (i.e., the last) column (i.e., iMax) of the 
     *----implicit ADF line cell to be rasterized (note the exclusion of topmost 
     *----edges and vertices falling exactly on a column)
     */
    iMax = I1616_TO_I32(xMax);
    if (iMax >= w) 
        iMax = w - 1;
    else if (xMax == I32_TO_I1616(iMax)) 
        iMax -= 1; 

    if (wnb)
    {
        /**
        *----This is a closed loop contour path for which a winding number buffer and
        *----boundary cell buffer exist. 
        */

        /**
        *----Determine the distance d from the line segment represented by the implicit
        *----ADF line cell to the bottommost (i.e., jMin) scanline to be rasterized
        */
        if ( (sampleSpacingY == I1616_CONST_1))
            d *= (ys - I32_TO_I1616(jMin));
        else
            d *= (ys - I1616_MUL(I32_TO_I1616(jMin), sampleSpacingY));

        /**
        *----Rasterize the implicit ADF horizontal line cell from bottom to top (i.e.,
        *----jMin to jMax), left to right (i.e., iMin to iMax). Assume ink on the right.
        *----For each scanline,determine the signed distance from the line segment to 
        *----the scanline and combine this distance with the corresponding distance in
        *----the distance buffer according to whether the sample point is
        *----inside a contour or outside as determined by the winding number buffer.
        *----Since the line segment is horizontal, the distance is
        *----constant for all samples in a scanline.
        *----Handle overlapping interior boundary points (winding number > 1) by first
        *----storing the negative of the distance in the boundary cell buffer to indicate
        *----the first boundary distance value. Later combine that distance with a
        *----distance value from a subsequent overlapping boundary, if such exists.
        *----Treat overlapping corner boundary cells as a single boundary cell value.
        */
        for (j = jMin; j <= jMax; j++) {

            ADF_I32 sample = j * w + iMin;

            /**
            *----Set a pointer to the entry in the distance buffer corresponding to the
            *----first sample point iMin of the current scanline
            */
            pDist    = &dist[sample];
            pWinding = &wnb[sample];
            if (bcell)
                pBcell   = &bcell[sample];

            /**
            *----Process each sample in the current scanline
            */
            for (i = iMin; i <= iMax; i++) {

                /**
                *----Combine the current line distance with its corresponding distance
                *----in the distance buffer
                */
                if (*pWinding == 1)
                {
                    if ((d > 0) && (d < *pDist)) 
                        *pDist = d;  /* inside pixel */
                }
                else if (*pWinding == 0)
                {
                    if ((d < 0) && (d > *pDist)) 
                        *pDist = d;  /* outside pixel */
                }
                else /* overlapping contours */
                {
                    if (pBcell && d > 0)
                    {
                        /**
                        *----Overlapping boundaries. 
                        */
                        if (*pBcell == 0)
                        {
                            /**
                            *----This is the first visit to this pixel and the
                            *----winding number is greater than one. We may visit
                            *----this pixel again, so store the current distance
                            *----in the boundary cell buffer in case we do, but do not set
                            *----the distance buffer because this may be an interior 
                            *----pixel with only one visit.
                            */
                            *pBcell = d;
                        }
                        else if (*pBcell > 0)
                        {
                            /**
                            *----A previous overlapping boundary cell point was set.
                            *----This is the second or higher visit to this pixel.
                            *----We have three choices for setting the distance.
                            *----These are (d,*pBcell,*pDist). We want to take
                            *----the distance in the middle and keep the smaller
                            *----value in bcell in case there are subsequent visits.
                            *----Ultimately, this process picks the larger of the
                            *----two smallest distances.
                            *----Note: on the second visit to this pixel,
                            *----*pDist==LARGE_INSIDE_DISTANCE so *pDist will always
                            *----be set to the maximum of (d,*pBcell) and *pBcell will
                            *----be set to the minimum of (d,*pBcell).
                            */
                            if(d > *pBcell)
                            {
                                if(d < *pDist)
                                    *pDist = d;
                            }
                            else
                            {
                                *pDist = *pBcell;
                                *pBcell = d;
                            }
                        }
                    } /* if (bcell) */
                }
                /**
                *----Update the pointers to the next entry in the buffers
                */
                pDist++;
                pWinding++;
                if (pBcell)
                    pBcell++;
            }

            /**
            *----Update the distance value to reflect the distance between the line
            *----segment and the next scanline
            */
            d += dIncrement;

            xMin += ddx_min;
            xMax += ddx_max;

            iMin = I1616_TO_I32(xMin);

            if (xMin < 0) iMin = 0;
            else if (xMin != I32_TO_I1616(iMin)) iMin += 1;

            iMax = I1616_TO_I32(xMax);
            if (iMax >= w)
                iMax = w - 1;
            else if (xMax == I32_TO_I1616(iMax))
                iMax -= 1;
        }
    } /* if ( wnb ) */
    else
    {

        /**
        *----This is a uniform stroke path for which no winding number buffer or
        *----boundary cell buffer exists.
        */

        /**
        *----Determine the distance d from the line segment represented by the implicit
        *----ADF line cell to the bottommost (i.e., jMin) scanline to be rasterized
        */
        if ( (sampleSpacingY == I1616_CONST_1))
            d = ys - I32_TO_I1616(jMin);
        else
            d = (ys - I1616_MUL(I32_TO_I1616(jMin), sampleSpacingY));

        /**
        *----Rasterize the implicit ADF horizontal line cell from bottom to top (i.e.,
        *----jMin to jMax), left to right (i.e., iMin to iMax). For each scanline,
        *----determine the unsigned distance from the line segment to the scanline and
        *----combine the negative of this distance with the corresponding distance in
        *----the distance buffer. Since the line segment is horizontal, the distance is
        *----constant for all samples in a scanline.
        */

        for (j = jMin; j <= jMax; j++) {


            /**
            *----Determine the negative of the magnitude of the signed distance from the
            *----current scanline to the line segment. This value is constant for all
            *----samples in the scanline.
            */
            FS_CONST ADF_I1616 dNeg = I1616_NEGABS(d);


            /**
            *----Set a pointer to the entry in the distance buffer corresponding to the
            *----first sample point iMin of the current scanline
            */
            pDist = &dist[j * w + iMin];


            /**
            *----Process each sample in the current scanline
            */
            for (i = iMin; i <= iMax; i++) {


                /**
                *----Combine the negative distance with its corresponding distance in
                *----the distance buffer
                */
                if (dNeg > *pDist) *pDist = dNeg;


                /**
                *----Update the pointer to the next entry in the distance buffer
                */
                pDist++;
            }


            /**
            *----Update the distance value to reflect the distance between the line
            *----segment and the next scanline
            */
            d -= sampleSpacingY;
        }
    } /* else no wnb */
}

/**
 *-----------------------------------------------------------------------------------
 *    Process an implicit ADF line cell for the line segment whose endpoints and unit
 *    normal vector are stored in the specified cell data (i.e., CellData). Rasterize
 *    the implicit ADF line cell scanline by scanline, in bottom to top, left to right
 *    order. For each sample point inside the implicit ADF line cell, compute the
 *    signed distance to the line segment and combine the distance with the corresponding
 *    distance stored in the distance buffer by choosing the distance with the
 *    smallest magnitude.
 *
 *    Horizonal and vertical line segments are identified and processed separately
 *    using functions with optimized implementations.
 *
 *    Overview
 *
 *        - Determine if the line segment is horizontal, vertical, or diagonal (i.e.,
 *          neither horizontal nor vertical).
 *        - Process vertical and horizontal line segments using
 *          ProcessVerticalLineCell() and ProcessHorizontalLineCell(), respectively,
 *          whose implementations are optimized for these special cases.
 *        - Given a diagonal line segment with endpoints (x0, y0) and (x1, y1) and unit
 *          normal vector (nx, ny), generate a rectangular implicit ADF line cell,
 *          where the center line of the rectangle is the line segment and the width of
 *          the rectangle is twice the filter radius, i.e., 2* filterRad. Determine the
 *          positions of the four vertices of the implicit ADF line cell by offsetting
 *          each endpoint in the directions (nx, ny) and (-nx, -ny) by a distance equal
 *          to filterRad. Assign a signed distance value to each of these 4 vertices,
 *          where the magnitude of the distance is equal to the distance from the
 *          vertex to the line segment, the sign of the distance value is chosen so
 *          that distances on opposite sides of the line segment have opposite signs,
 *          and distances increase (i.e., become more positive) from left to right
 *          along a horizontal scanline.
 *        - Order the cell vertices in a clockwise direction starting from the
 *          bottommost vertex.
 *        - Determine the cells two left edges and two right edges from the ordered
 *          vertices and set parameters required for rasterizing the edges and for
 *          interpolating the signed distance field between left and right edges of the
 *          rectangle.
 *        - Order the left and right edges from bottom to top in left and right edge
 *          lists, respectively.
 *        - Rasterize the implicit ADF line cell scanline by scanline, from bottom to
 *          top, left to right, using an active ordered edge list algorithm. When
 *          rasterizing each scanline, use a digital differential analyzer to
 *          interpolate a signed distance for each interior sample point between the
 *          current left and right edges of the active edge list from the distance
 *          values computed at cell vertices. Combine the distance with the corresponding negative
 *          distance in the distance buffer by taking the value with the smallest 
 *          magnitude.
 *
 *    Implementation Notes
 *
 *        - The rectangular geometry of an implicit ADF line cell is constructed such 
 *          that it exactly covers the area obtained by sweeping a line L along the 
 *          line segment, where L is perpendicular to the line segment and extends on 
 *          each side of the line segment by filterRad. For this reason, an implicit 
 *          ADF line cell has optimal geometry, i.e., it is (reasonably) fast to 
 *          generate and it encloses only those sample points necessary to process the 
 *          line segment.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Data required to represent each edge of an implicit ADF line cell (for
 *    rasterization) is stored in a LineEdge data structure. LineEdge contains the
 *    following elements:
 *
 *    yTop: The y-coordinate of the topmost endpoint of the edge.
 *
 *    x: The x-intercept where the edge crosses the current scanline. This value is 
 *    initialized to the x-intercept where the edge crosses the bottommost scanline 
 *    that it intersects.
 *
 *    dxy: The change in the x-intercept x with a unit increase in y. This value is  
 *    used to increment the edge's x-intercept when the rasterizer advances to the 
 *    next scanline when proceeding from bottom to top.
 *
 *    d: The distance from the current x-intercept to the line segment represented by 
 *    the implicit ADF line cell. This value is initialized to the distance from the 
 *    initial x-intercept to the line segment.
 *
 *    ddy: The change in the distance d with a unit increase in y. This value is used 
 *    to increment d when the rasterizer advances to the next scanline when proceeding 
 *    from bottom to top. The distance d can be incremented along edges of implicit ADF 
 *    line cells in this manner because the distance field is linear inside and on the 
 *    boundary of implicit ADF line cells.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
typedef struct {
    ADF_I1616 yTop; /* The y-coordinate of the topmost endpoint of the edge */
    ADF_I1616 x;    /* The x-intercept where the edge crosses the current scanline */
    ADF_I1616 dxy;  /* The change in x with a unit increase in y */
    ADF_I1616 d;    /* The distance from the current x-intercept to the line segment */
    ADF_I1616 ddy;  /* The change in d with a unit increase in y */
}    LineEdge;

/**
 *-----------------------------------------------------------------------------------
 * find the interset points, see fs_stik.c, where the calculation was based on FS_FIXED 
 * find the intersection point x,y of given a point <xs,ys> with a
 * unit tangent t0x,t0y, with another point <xe,ye> with unit tangent t1x, t1y
 *-----------------------------------------------------------------------------------
 */
static int intersect( ADF_I1616 xs, ADF_I1616 ys, ADF_I1616 xe, ADF_I1616 ye,
                      ADF_I1616 t0x, ADF_I1616 t0y, ADF_I1616  t1x, ADF_I1616  t1y,
                      ADF_I1616  *x, ADF_I1616 *y)
{
    ADF_I1616 a,b,s;
    ADF_I32 divStatus;

    /* <p0> and <p1> are unit tangents ... <b> is the cross product which is the 
    * sin of the included angle which is in the invertal [-FIXED_ONE ... FIXED_ONE] */
    b = I1616_MUL(t0x, t1y) - I1616_MUL(t0y, t1x);

    /* unit tangents are almost parallel ... intersection will be bizarre
    * just use the midpoint of the chord from (x0,y0) to (x1,y1) */

    if (ABS(b) <= SIN_2_DEGREES)
    {
        *x = (xs + xe)/2;
        *y = (ys + ye)/2;
        return 1;
    }

    /* now we know we can subtract any two coordinates safely, and we're
     * multiplying by a value in the interval [-FIXED_ONE ... FIXED_ONE] so
     * there is no chance of overflow on <a> */
    a = I1616_MUL(t1y, xe - xs) - I1616_MUL(t1x, ye - ys);

    /* No assurances about overflow here ... BUT in the usage this was designed
     * for (finding the control point for the left/right offset parabola's of a 
     * stroke) we should be OK */

    s = I1616_DIV(a,b, &divStatus);
    *x = xs + I1616_MUL(s,t0x);
    *y = ys + I1616_MUL(s,t0y);

    /* return OK if (x,y) aren't at infinity */
    return (ABS(*x)!=MYINFINITY && ABS(*y)!=MYINFINITY) ? 0 : 1;
}


/**
 *-----------------------------------------------------------------------------------
 * Set the properties of the line cell trapezoid's left and right edges. There are
 * sixteen possible situations (four cases and four possibilities in each case).
 *-----------------------------------------------------------------------------------
 */
static ADF_Void Set_Left_Right_Edges(CellData *cell,
                                     ADF_I1616 *x, ADF_I1616 *y, ADF_U32 *idx,
                                     ADF_I1616 dxyPara, ADF_I1616 dxy_pre, ADF_I1616 dxy_next,
                                     ADF_I1616 bi_pre_tx,  ADF_I1616 bi_pre_ty,
                                     ADF_I1616 bi_next_tx, ADF_I1616 bi_next_ty,
                                     ADF_I1616 cos_pre, ADF_I1616 cos_next,
                                     ADF_I1616 invSampleSpacingX, ADF_I1616 sampleSpacingY,
                                     ADF_I1616 invSampleSpacingY,
                                     ADF_I1616 filterRad,
                                     LineEdge *left, LineEdge *right,
                                     ADF_U32 *num_lefts, ADF_U32 *num_rights)
{
    ADF_I1616 nx = (cell->nx);
    ADF_I1616 ny = (cell->ny);
    ADF_I1616 ys = (cell->y0);
    ADF_I1616 ye = (cell->y1);
    ADF_I1616 xs = (cell->x0);
    ADF_I1616 xe = (cell->x1);
    ADF_I1616 yMin;
    ADF_I32   jMin;

    ADF_I1616 tx = nx, ty = ny;
    ADF_I1616 dy0, dy1;
    ADF_I1616 dy2 = 0;
    ADF_I1616 d0 = 0, d1 = 0, ddy0, ddy1 = 0;
    ADF_I1616 d2 = 0, ddy2 = 0;
    ADF_I1616 dxyl0 = 0, dxyl1 = 0, dxyl2 = 0, dxyr0, dxyr1 = 0, dxyr2 = 0;
    ADF_I32 divStatus, total_edges;

    if ( ys <= ye ) /* bottom to top */
    {
        /* case 1: */
        int min = 0;
        if (( y[3] <= y[0] ) || (y[2] <= y[1]))
        {
            ADF_I1616 xx = 0, yy = 0;
            int max = 0;
            int onRight = 0;
            ADF_I1616 cy[3] = {0};
            ADF_I1616 cx[3] = {0};
            if ( (y[2] == y[1]) && x[2] == x[1])
            {
                cy[2] = y[2];
                cx[2] = x[2];
                y[2] = y[3];
                x[2] = x[3];
            }
            else
            {
                /*ADF_I1616 bi_pre_tx_save = bi_pre_tx;
                ADF_I1616 bi_next_tx_save = bi_next_tx;*/ /* not needed per Lint */
                ADF_I1616 bi_pre_ty_save = bi_pre_ty;
                ADF_I1616 bi_next_ty_save = bi_next_ty;
                if (invSampleSpacingX != I1616_CONST_1)
                {
                    bi_pre_tx = I1616_MUL(bi_pre_tx, invSampleSpacingX);
                    bi_next_tx = I1616_MUL(bi_next_tx, invSampleSpacingX);
                }
                if (invSampleSpacingY != I1616_CONST_1)
                {
                    bi_pre_ty = I1616_MUL(bi_pre_ty, invSampleSpacingY);
                    bi_next_ty = I1616_MUL(bi_next_ty, invSampleSpacingY);
                }
                I1616_NORMALIZE(bi_pre_tx, bi_pre_ty, &bi_pre_tx, &bi_pre_ty);
                I1616_NORMALIZE(bi_next_tx, bi_next_ty, &bi_next_tx, &bi_next_ty);

                intersect(x[1], y[1], x[2], y[2], bi_pre_tx, bi_pre_ty,
                            bi_next_tx,bi_next_ty, &xx, &yy);

                /*bi_pre_tx = bi_pre_tx_save;
                bi_next_tx = bi_next_tx_save;*/
                bi_pre_ty = bi_pre_ty_save;
                bi_next_ty = bi_next_ty_save;

                if (y[3] < y[0])
                {
                    onRight = 1;
                    cx[0] = x[0];
                    x[0] = xx;
                    cy[0] = y[0];
                    y[0] = yy;
                    if (y[0] < y[1])
                    {
                        min = 0;
                        max = 2;
                    }
                    else
                    {
                        min = 1;
                        if (y[0] < y[2])
                            max = 2;
                        else
                            max = 0;
                    }
                }
                else
                {
                    cx[1] = x[1];
                    x[1] = xx; 
                    cy[1] = y[1];
                    y[1] = yy;
                    cx[2] = x[2];
                    x[2] = x[3];
                    cy[2] = y[2];
                    y[2] = y[3];
                    if ( y[1] < y[0])
                    {
                        min = 1;
                        max = 2;
                    }
                    else
                    {
                        min = 0;
                        if (y[1] > y[2])
                            max = 1;
                        else
                            max = 2;
                    }
                }
            }
            if (min == 0)
            {
                idx[0] = 0; idx[1] = 1; idx[2] = 2;
                dxyl0 = dxy_pre;
                if (cos_pre <= COS_CHECK)
                    ddy0 = I1616_DIV(sampleSpacingY, ny, &divStatus);
                else
                {
                    ddy0 = ty + I1616_MUL(dxy_pre, tx);
                    ddy0 = ABS(ddy0);
                    if (bi_pre_ty < 0)
                        ddy0 = -ddy0;
                    if ( (sampleSpacingY != I1616_CONST_1))
                        ddy0 = I1616_MUL(sampleSpacingY,ddy0);
                }
                if (max == 2)
                {
                    idx[4] = 2;
                    *num_lefts = 2;
                    *num_rights = 1;
                    if (onRight)
                    {
                        dxyl1 = dxyPara;
                        dxyr0 = dxy_next;
                        if (invSampleSpacingX != I1616_CONST_1)
                            xe = I1616_MUL(xe, invSampleSpacingX);
                        if (invSampleSpacingY != I1616_CONST_1)
                            ye = I1616_MUL(ye, invSampleSpacingY);

                        if ( ABS(ye - yy) > ABS(xe - xx) )
                            d0 = I1616_DIV(ye - yy, ye - y[3], &divStatus);
                        else
                            d0 = I1616_DIV(xe - xx, xe - x[3], &divStatus);

                        d0 = ABS(d0);
                        d0 = I1616_MUL(d0, filterRad);
                        d1 = I1616_NEGATE(filterRad); 
                        ddy1 = 0;
                    }
                    else
                    {
                        dxyl1 = dxy_next;
                        dxyr0 = dxyPara;
                        d0 = filterRad;
                        if (invSampleSpacingX != I1616_CONST_1)
                            xs = I1616_MUL(xs, invSampleSpacingX);
                        if (invSampleSpacingY != I1616_CONST_1)
                            ys = I1616_MUL(ys, invSampleSpacingY);

                        if (ABS(yy - ys) > ABS(xx - xs))
                            d1 = I1616_DIV(yy - ys, cy[1] - ys, &divStatus);
                        else
                            d1 = I1616_DIV(xx - xs, cx[1] - xs, &divStatus);

                        d1 = ABS(d1);
                        d1 = -I1616_MUL(d1, filterRad);
                        if (cos_next <= COS_CHECK)
                            ddy1 = I1616_DIV(sampleSpacingY, ny, &divStatus);
                        else
                        {
                            ddy1 = ty + I1616_MUL(dxy_next,tx);
                            ddy1 = ABS(ddy1);
                            if (bi_next_ty < 0)
                                ddy1 = -ddy1;
                            if ((sampleSpacingY != I1616_CONST_1))
                                ddy1 = I1616_MUL(sampleSpacingY,ddy1);
                        }
                    }
                }
                else /* max = 1 */
                {
                    idx[4] = 1;
                    *num_lefts = 1;
                    *num_rights = 2;
                    dxyr0 = dxyPara;
                    dxyr1 = dxy_next;
                    d0 = filterRad;
                }
            }
            else /* if min = 1*/
            {
                idx[0] = 1; idx[1] = 2; idx[2] = 0;
                if (max == 2)
                {
                    idx[4] = 2;
                    *num_lefts = 1;
                    *num_rights = 2;
                    dxyr0 = dxy_pre;
                    if (onRight)
                    {
                        dxyl0 = dxyPara;
                        dxyr1 = dxy_next;
                        d0 = I1616_NEGATE(filterRad);
                        ddy0 = 0;
                    }
                    else
                    {
                        dxyl0 = dxy_next;
                        dxyr1 = dxyPara;

                        if (invSampleSpacingX != I1616_CONST_1)
                            xe = I1616_MUL(xe, invSampleSpacingX);
                        if (invSampleSpacingY != I1616_CONST_1)
                            ye = I1616_MUL(ye, invSampleSpacingY);

                        if (ABS(ye - yy) > ABS(xe - xx))
                            d0 = I1616_DIV(ye - yy, ye - cy[2], &divStatus);
                        else
                            d0 = I1616_DIV(xe - xx, xe - cx[2], &divStatus);
                        d0 = ABS(d0);
                        d0 = -I1616_MUL(d0, filterRad);
                        if (cos_next <= COS_CHECK) 
                            ddy0 = I1616_DIV(sampleSpacingY, ny, &divStatus);
                        else
                        {
                            ddy0 = ty + I1616_MUL(dxy_next,tx);
                            ddy0 = ABS(ddy0);
                            if (bi_next_ty < 0)
                                ddy0 = -ddy0;
                            if ( (sampleSpacingY != I1616_CONST_1))
                                ddy0 = I1616_MUL(sampleSpacingY,ddy0);
                        }
                    }
                }
                else
                {
                    idx[4] = 0;
                    *num_lefts = 2;
                    *num_rights = 1;
                    dxyl0 = dxyPara;;
                    dxyl1 = dxy_next;
                    dxyr0 = dxy_pre;
                    d0 = I1616_NEGATE(filterRad);
                    d1 = d0;
                    ddy0 = 0;
                    if (cos_next <= COS_CHECK) 
                        ddy1 = I1616_DIV(sampleSpacingY, ny, &divStatus);
                    else
                    {
                        ddy1 = ty + I1616_MUL(dxy_next,tx);
                        ddy1 = ABS(ddy1);
                        if (bi_next_ty < 0)
                            ddy1 = -ddy1;
                        if ( (sampleSpacingY != I1616_CONST_1))
                            ddy1 = I1616_MUL(sampleSpacingY,ddy1);
                    }
                }
            } /* else if min = 1 */
        } /* if (( y[3] <= y[0] ) || (y[2] <= y[1])) */
        else 
        {
            /* case 2: */
            /* we know that in this case y[3] > y[0], y[2] > y[1] */
            if ( y[0] < y[1] )
                min = 0;
            else
                min = 1;

            if ( min == 0 )
            {
                idx[0] = 0; idx[1] = 1; idx[2] = 2; idx[3] = 3;
                dxyl0 = dxy_pre;
                dxyl1 = dxyPara;
                dxyr0 = dxyPara;
                d0 =  filterRad;
                d1 =  I1616_NEGATE(filterRad);
                if (cos_pre <= COS_CHECK) 
                    ddy0 = I1616_DIV(sampleSpacingY, ny, &divStatus);
                else
                {
                    ddy0 = ty + I1616_MUL(dxy_pre, tx);
                    ddy0 = ABS(ddy0);
                    if (bi_pre_ty < 0)
                        ddy0 = -ddy0;
                    if ( (sampleSpacingY != I1616_CONST_1))
                        ddy0 = I1616_MUL(sampleSpacingY,ddy0);
                }
                ddy1 = 0;

                if (y[2] < y[3])
                {
                    idx[4] = 3; 
                    *num_lefts = 3;
                    *num_rights = 1;
                    dxyl2 = dxy_next;
                    d2 = d1;
                    if (cos_next <= COS_CHECK)
                        ddy2 = I1616_DIV(sampleSpacingY, ny, &divStatus);
                    else
                    {
                        ddy2 = ty + I1616_MUL(dxy_next,tx);
                        ddy2 = ABS(ddy2);
                        if (bi_next_ty < 0)
                            ddy2 = -ddy2;
                        if ( (sampleSpacingY != I1616_CONST_1))
                            ddy2 = I1616_MUL(sampleSpacingY,ddy2);
                    }
                }
                else
                {
                    idx[4] = 2;
                    *num_lefts = 2;
                    *num_rights = 2;
                    dxyr1 = dxy_next;
                }
            }
            else
            {
                idx[0] = 1; idx[1] = 2; idx[2] = 3; idx[3] = 0;
                dxyl0 = dxyPara;
                dxyr0 = dxy_pre;
                dxyr1 = dxyPara;
                d0 = I1616_NEGATE(filterRad);
                ddy0 = 0;
                if (y[2] < y[3])
                {
                    idx[4] = 3;
                    *num_lefts = 2;
                    *num_rights = 2;
                    dxyl1 = dxy_next;
                    d1 = d0;
                    if (cos_next <= COS_CHECK) 
                        ddy1 = I1616_DIV(sampleSpacingY, cell->ny, &divStatus);
                    else
                    {
                        ddy1 = ty + I1616_MUL(dxy_next,tx);
                        ddy1 = ABS(ddy1);
                        if (bi_next_ty < 0)
                            ddy1 = -ddy1;
                        if ( (sampleSpacingY != I1616_CONST_1))
                            ddy1 = I1616_MUL(sampleSpacingY,ddy1);
                    }
                }
                else
                {
                    idx[4] = 2;
                    *num_lefts = 1;
                    *num_rights = 3;
                    dxyr2 = dxy_next;
                }
            }
        }
    } /* endof if ( ys <= y1 ) */
    else
    {
        /* case 3: */
        int min = 0;
        if (( y[0] <= y[3] ) || (y[1] <= y[2]))
        {
            ADF_I1616 xx = 0, yy = 0;
            int max = 0;
            int onRight = 0;
            ADF_I1616 cy[3] = {0};
            ADF_I1616 cx[3] = {0};
            if ( (y[2] == y[1]) && x[2] == x[1])
            {
                cy[2] = y[2];
                cx[2] = x[2];
                y[2] = y[3];
                x[2] = x[3];
            }
            else
            {
                /*ADF_I1616 bi_pre_tx_save = bi_pre_tx;
                ADF_I1616 bi_next_tx_save = bi_next_tx;*/
                ADF_I1616 bi_pre_ty_save = bi_pre_ty;
                ADF_I1616 bi_next_ty_save = bi_next_ty;

                if (invSampleSpacingX != I1616_CONST_1)
                {
                    bi_pre_tx = I1616_MUL(bi_pre_tx, invSampleSpacingX);
                    bi_next_tx = I1616_MUL(bi_next_tx, invSampleSpacingX);
                }
                if (invSampleSpacingY != I1616_CONST_1)
                {
                    bi_pre_ty = I1616_MUL(bi_pre_ty, invSampleSpacingY);
                    bi_next_ty = I1616_MUL(bi_next_ty, invSampleSpacingY);
                }
                I1616_NORMALIZE(bi_pre_tx, bi_pre_ty, &bi_pre_tx, &bi_pre_ty);
                I1616_NORMALIZE(bi_next_tx, bi_next_ty, &bi_next_tx, &bi_next_ty);

                intersect(x[1], y[1], x[2], y[2], bi_pre_tx, bi_pre_ty,
                            bi_next_tx,bi_next_ty, &xx, &yy);

                /*bi_pre_tx = bi_pre_tx_save;
                bi_next_tx = bi_next_tx_save;*/
                bi_pre_ty = bi_pre_ty_save;
                bi_next_ty = bi_next_ty_save;


                if (y[0] < y[3])
                {
                    onRight = 1;
                    cx[0] = x[0];
                    x[0] = xx; 
                    cy[0] = y[0];
                    y[0] = yy;
                    if (y[0] < y[2]) /* max = 1 */
                        min = 0;
                    else
                    {
                        min = 2;
                        if (y[0] > y[1])
                            max = 0;
                        else
                            max = 1;
                    }
                }
                else 
                {
                    cx[1] = x[1];
                    x[1] = xx; 
                    cy[1] = y[1];
                    y[1] = yy;
                    cx[2] = x[2];
                    x[2] = x[3];
                    cy[2] = y[2];
                    y[2] = y[3];
                    if (y[1] < y[2]) /* max = 0 */
                        min = 1;
                    else
                    {
                        min = 2;
                        if (y[1] > y[0])
                            max = 1;
                        else
                            max = 0;
                    }
                }
            }

            if (min == 0) /* max = 1 */
            {
                idx[0] = 0; idx[1] = 1; idx[2] = 2;
                idx[4] = 1;
                *num_lefts = 1;
                *num_rights = 2;
                dxyl0 = dxy_pre;
                dxyr0 = dxy_next;
                dxyr1 = dxyPara;
                if (invSampleSpacingX != I1616_CONST_1)
                    xs = I1616_MUL(xs, invSampleSpacingX);
                if (invSampleSpacingY != I1616_CONST_1)
                    ys = I1616_MUL(ys, invSampleSpacingY);

                if (ABS(ys - yy) > ABS(xs - xx))
                    d0 = I1616_DIV(ys - yy, ys - cy[0], &divStatus);
                else
                    d0 = I1616_DIV(xs - xx, xs - cx[0], &divStatus);
                d0 = ABS(d0);
                d0 = I1616_MUL(d0, filterRad);
                if (cos_pre <= COS_CHECK)
                    ddy0 = I1616_DIV(sampleSpacingY, ny, &divStatus);
                else
                {
                    ddy0 = ty + I1616_MUL(dxy_pre, tx);
                    ddy0 = ABS(ddy0);
                    if (bi_pre_ty < 0)
                        ddy0 = -ddy0;
                    if ( (sampleSpacingY != I1616_CONST_1))
                        ddy0 = I1616_MUL(sampleSpacingY,ddy0);
                }
            }
            else if (min == 1) /* max = 0 */
            {
                    idx[0] = 1; idx[1] = 2; idx[2] = 0;
                    idx[4] = 0; 
                    *num_lefts = 2;
                    *num_rights = 1;
                    dxyl0 = dxy_next;
                    dxyl1 = dxyPara;
                    dxyr0 = dxy_pre;

                    if (invSampleSpacingX != I1616_CONST_1)
                        xs = I1616_MUL(xs, invSampleSpacingX);
                    if (invSampleSpacingY != I1616_CONST_1)
                        ys = I1616_MUL(ys, invSampleSpacingY);

                    if (ABS(ys - yy) > ABS(xs - xx))
                        d0 = I1616_DIV(ys - yy, ys - cy[1], &divStatus);
                    else
                        d0 = I1616_DIV(xs - xx, xs - cx[1], &divStatus);
                    d0 = ABS(d0);
                    d0 = -I1616_MUL(d0, filterRad);
                    d1 = filterRad;
                    if (cos_next <= COS_CHECK)
                        ddy0 = I1616_DIV(sampleSpacingY, cell->ny, &divStatus);
                    else
                    {
                        ddy0 = ty + I1616_MUL(dxy_next,tx);
                        ddy0 = ABS(ddy0);
                        if (bi_next_ty < 0)
                            ddy0 = -ddy0;
                        if ((sampleSpacingY != I1616_CONST_1))
                            ddy0 = I1616_MUL(sampleSpacingY,ddy0);
                    }
                    ddy1 = 0;
            }
            else /* min = 2 */
            {
                idx[0] = 2; idx[1] = 0; idx[2] = 1;
                if (max == 0)
                {
                    idx[4] = 0;
                    *num_lefts = 1;
                    *num_rights = 2;
                    if (onRight)
                    {
                        dxyl0 = dxy_next;
                        dxyr0 = dxyPara;
                        dxyr1 = dxy_pre;
                        d0 = -filterRad;
                        if (cos_next <= COS_CHECK)
                            ddy0 = I1616_DIV(sampleSpacingY, cell->ny, &divStatus);
                        else
                        {
                            ddy0 = ty + I1616_MUL(dxy_next,tx);
                            ddy0 = ABS(ddy0);
                            if (bi_next_ty < 0)
                                ddy0 = -ddy0;
                            if ( (sampleSpacingY != I1616_CONST_1))
                                ddy0 = I1616_MUL(sampleSpacingY,ddy0);
                        }
                    }
                    else
                    {
                        dxyl0 = dxyPara;
                        dxyr0 = dxy_next;
                        dxyr1 = dxy_pre;
                        d0 = filterRad;
                        ddy0 = 0;
                    }
                }
                else /* max = 1*/
                {
                    idx[4] = 1; 
                    *num_lefts = 2;
                    *num_rights = 1;
                    if (cos_pre <= COS_CHECK)
                        ddy1 = I1616_DIV(sampleSpacingY, ny, &divStatus);
                    else
                    {
                        ddy1 = ty + I1616_MUL(dxy_pre, tx);
                        ddy1 = ABS(ddy1);
                        if (bi_pre_ty < 0)
                            ddy1 = -ddy1;
                        if ( (sampleSpacingY != I1616_CONST_1))
                            ddy1 = I1616_MUL(sampleSpacingY,ddy1); 
                    }
                    if (onRight)
                    {
                        dxyl0 = dxy_next;
                        dxyl1 = dxy_pre;
                        dxyr0 = dxyPara;
                        d0 = I1616_NEGATE(filterRad);
                        if (invSampleSpacingX != I1616_CONST_1)
                            xe = I1616_MUL(xe, invSampleSpacingX);
                        if (invSampleSpacingY != I1616_CONST_1)
                            ye = I1616_MUL(ye, invSampleSpacingY);

                        if (ABS(yy - ye) > ABS(xx - xe))
                            d1 = I1616_DIV(yy - ye, y[3] - ye, &divStatus);
                        else
                            d1 = I1616_DIV(xx - xe, x[3] - xe, &divStatus);

                        d1 = ABS(d1);
                        d1 = I1616_MUL(d1, filterRad);
                        if (cos_next <= COS_CHECK)
                            ddy0 = I1616_DIV(sampleSpacingY, cell->ny, &divStatus);
                        else
                        {
                            ddy0 = ty + I1616_MUL(dxy_next,tx);
                            ddy0 = ABS(ddy0);
                            if (bi_next_ty < 0)
                                ddy0 = -ddy0;
                            if ( (sampleSpacingY != I1616_CONST_1))
                                ddy0 = I1616_MUL(sampleSpacingY,ddy0);
                        }
                    }
                    else
                    {
                        dxyl0 = dxyPara;
                        dxyl1 = dxy_pre;
                        dxyr0 = dxy_next;
                        d0 = filterRad;
                        d1 = filterRad;
                        ddy0 = 0;
                    }
                }
            }
        } /* if (( y[0] <= y[3] ) || (y[1] <= y[2])) */
        else
        {
            /* case 4: */
            /* we know that in this case y[0] > y[3], y[1] > y[2] */
            if ( y[3] < y[2] )
                min = 3;
            else
                min = 2;
            if ( min == 2)
            {
                idx[0] = 2; idx[1] = 3; idx[2] = 0; idx[3] = 1;
                dxyl0 = dxy_next;
                dxyl1 = dxyPara;
                dxyr0 = dxyPara;
                d0 = I1616_NEGATE(filterRad);
                if (cos_next <= COS_CHECK)
                    ddy0 = I1616_DIV(sampleSpacingY, cell->ny, &divStatus);
                else
                {
                    ddy0 = ty + I1616_MUL(dxy_next,tx);
                    ddy0 = ABS(ddy0);
                    if (bi_next_ty < 0)
                        ddy0 = -ddy0;
                    if ( (sampleSpacingY != I1616_CONST_1))
                        ddy0 = I1616_MUL(sampleSpacingY,ddy0);
                }
                ddy1 = 0;
                if (y[0] < y[1])
                { 
                    idx[4] = 1;
                    *num_lefts = 3;
                    *num_rights = 1;
                    dxyl2 = dxy_pre;
                    d1 = d2 = filterRad;
                    if (cos_pre <= COS_CHECK)
                        ddy2 = I1616_DIV(sampleSpacingY, ny, &divStatus);
                    else
                    {
                        ddy2 = ty + I1616_MUL(dxy_pre, tx);
                        ddy2 = ABS(ddy2);
                        if (bi_pre_ty < 0)
                            ddy2 = -ddy2;
                        if ( (sampleSpacingY != I1616_CONST_1))
                            ddy2 = I1616_MUL(sampleSpacingY,ddy2);
                    }
                }
                else
                {
                    idx[4] = 0;
                    *num_lefts = 2;
                    *num_rights = 2;
                    dxyr1 = dxy_pre;
                    d1 = filterRad;
                }
            }
            else /* min == 3 */
            {
                idx[0] = 3; idx[1] = 0; idx[2] = 1; idx[3] = 2;
                dxyl0 = dxyPara;
                dxyr0 = dxy_next;
                dxyr1 = dxyPara;
                d0 = filterRad; 
                ddy0 = 0;  
                if (y[0] < y[1]) 
                {
                    idx[4] = 1;
                    *num_lefts = 2;
                    *num_rights = 2;
                    dxyl1 = dxy_pre;
                    d1 = filterRad; 
                    if (cos_pre <= COS_CHECK) 
                        ddy1 = I1616_DIV(sampleSpacingY, ny, &divStatus);
                    else
                    {
                        ddy1 = ty + I1616_MUL(dxy_pre, tx);
                        ddy1 = ABS(ddy1);
                        if (bi_pre_ty < 0)
                            ddy1 = -ddy1;
                        if ( (sampleSpacingY != I1616_CONST_1))
                            ddy1 = I1616_MUL(sampleSpacingY,ddy1);
                    }
                }
                else
                {
                    idx[4] = 0;
                    *num_lefts = 1;
                    *num_rights = 3;
                    dxyr2 = dxy_pre; 
                }
            }
        }
    } /* top to bottom */

    total_edges = *num_lefts + *num_rights;
    if (total_edges > 4)
        return; /* should never happen */

    /**
     *----Determine the bottommost (i.e., the first) scanline (i.e., jMin) of the
     *----implicit ADF line cell to be rasterized (note the inclusion of
     *----bottommost vertices falling exactly on a scanline).
     */
    yMin = y[idx[0]];
    jMin = I1616_TO_I32(yMin);
    if (yMin < 0) jMin = 0;
    else if (yMin != I32_TO_I1616(jMin)) jMin += 1;


    /**
     *----Adjust dx/dy slope for subpixel sample spacing
     */
    if ( (invSampleSpacingX != I1616_CONST_1) )
        dxyl0 = I1616_MUL(dxyl0, invSampleSpacingX);

    if ( (sampleSpacingY != I1616_CONST_1) )
        dxyl0 = I1616_MUL(dxyl0, sampleSpacingY);


    /**
     *----Initialize the first left edge. dy0 is the height of jMin (i.e., the
     *----first scanline to be rasterized) above the bottommost vertex of the
     *----implicit ADF line cell; dy0 is used to determine the first x-intercept
     *----and the distance at the first x-intercept for the first left
     *----edge. Since this height is the same for the first right edge, dy0 is
     *----reused below when initializing the first right edge.
     */
    dy0 = I32_TO_I1616(jMin) - y[idx[0]];
    left[0].yTop = y[idx[1]];
    left[0].dxy = dxyl0;
    left[0].ddy = ddy0;
    left[0].x = x[idx[0]] + I1616_MUL(dy0, dxyl0);
    left[0].d = d0 + I1616_MUL(dy0, ddy0);


    /**
     *----Initialize the second left edge. dy1 is the height of the first
     *----scanline intersecting the second left edge above the bottommost vertex
     *----of the second left edge; dy1 is used to determine the first x-intercept
     *----and the distance at the first x-intercept for the second left edge.
     */
    if (*num_lefts >=2)
    {
        /**
         *----Adjust dx/dy slope for subpixel sample spacing
         */
        if ( (invSampleSpacingX != I1616_CONST_1) )
            dxyl1 = I1616_MUL(dxyl1, invSampleSpacingX);

        if ( (sampleSpacingY != I1616_CONST_1) )
            dxyl1 = I1616_MUL(dxyl1, sampleSpacingY);

        if (y[idx[1]] <= 0) dy1 = I1616_NEGATE(y[idx[1]]);
        else
        {
            dy1 = I32_TO_I1616(I1616_TO_I32(y[idx[1]])) - y[idx[1]];
            if (dy1 != 0) dy1 = dy1 + I1616_CONST_1;
        }
        left[1].yTop = y[idx[2]];
        left[1].dxy = dxyl1;
        left[1].ddy = ddy1;
        left[1].x = x[idx[1]] + I1616_MUL(dy1, dxyl1);
        left[1].d = d1 +I1616_MUL(dy1, ddy1);
    }

    if (*num_lefts ==3)
    {
        /**
         *----Adjust dx/dy slope for subpixel sample spacing
         */
        if ( (invSampleSpacingX != I1616_CONST_1) )
            dxyl2 = I1616_MUL(dxyl2, invSampleSpacingX);

        if ( (sampleSpacingY != I1616_CONST_1) )
            dxyl2 = I1616_MUL(dxyl2, sampleSpacingY);

        if (y[idx[2]] <= 0) dy2 = I1616_NEGATE(y[idx[2]]);
        else 
        {
            dy2 = I32_TO_I1616(I1616_TO_I32(y[idx[2]])) - y[idx[2]];    
            if (dy2 != 0) dy2 = dy2 + I1616_CONST_1;
        }
        left[2].yTop = y[idx[3]];
        left[2].dxy = dxyl2;
        left[2].ddy = ddy2;
        left[2].x = x[idx[2]] + I1616_MUL(dy2, dxyl2);
        left[2].d = d2 +I1616_MUL(dy2, ddy2);
    }


    /**
     *----Adjust dx/dy slope for subpixel sample spacing
     */
    if ( (invSampleSpacingX != I1616_CONST_1) )
        dxyr0 = I1616_MUL(dxyr0, invSampleSpacingX);

    if ( (sampleSpacingY != I1616_CONST_1) )
        dxyr0 = I1616_MUL(dxyr0, sampleSpacingY);


    /**
     *----Initialize the first right edge
     */
    right[0].yTop = y[idx[total_edges-1]];
    right[0].dxy = dxyr0;
    right[0].x = x[idx[0]] + I1616_MUL(dy0, dxyr0);

    /**
     *----Initialize the second right edge. dy1 is the height of the first
     *----scanline intersecting the second right edge above the bottommost vertex
     *----of the second right edge; dy1 is used to determine the first
     *----x-intercept and the distance at the first x-intercept for the second
     *----right edge.
     */

    if (*num_rights >=2)
    {
        /**
         *----Adjust dx/dy slope for subpixel sample spacing
         */
        if ( (invSampleSpacingX != I1616_CONST_1) )
            dxyr1 = I1616_MUL(dxyr1, invSampleSpacingX);

        if ( (sampleSpacingY != I1616_CONST_1) )
            dxyr1 = I1616_MUL(dxyr1, sampleSpacingY);

        if (y[idx[total_edges-1]] <= 0) 
            dy1 = I1616_NEGATE(y[idx[total_edges-1]]);
        else 
        {
            dy1 = I32_TO_I1616(I1616_TO_I32(y[idx[total_edges-1]])) - y[idx[total_edges-1]];
            if (dy1 != 0) 
                dy1 = dy1 + I1616_CONST_1;
        }
        right[1].yTop = y[idx[total_edges-2]];  
        right[1].dxy = dxyr1;
        right[1].x = x[idx[total_edges-1]] + I1616_MUL(dy1, dxyr1);
    }

    if (*num_rights == 3)
    {
        /**
         *----Adjust dx/dy slope for subpixel sample spacing
         */
        if ( (invSampleSpacingX != I1616_CONST_1) )
            dxyr2 = I1616_MUL(dxyr2, invSampleSpacingX);

        if ( (sampleSpacingY != I1616_CONST_1) )
            dxyr2 = I1616_MUL(dxyr2, sampleSpacingY);

        if (y[idx[total_edges-2]] <= 0) 
            dy1 = I1616_NEGATE(y[idx[total_edges-2]]);
        else
        {
            dy1 = I32_TO_I1616(I1616_TO_I32(y[idx[total_edges-2]])) - y[idx[total_edges-2]];
            if (dy1 != 0) 
                dy1 = dy1 + I1616_CONST_1;
        }
        right[2].yTop = y[idx[total_edges-3]];
        right[2].dxy = dxyr2;
        right[2].x = x[idx[total_edges-2]] + I1616_MUL(dy1, dxyr2);
    }
}


/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
static ADF_Void ProcessLineCell_trap (CellData *cell, DistBuffer *distBuf, ADF_I1616
        filterRad, ADF_I8 *wnb, ADF_I1616 *bcell, int start, ADF_I1616* p_cos_pre,
        ADF_I1616* p_xyt)
{

    /**
     *----This line segment is diagonal or vertical. Construct an implicit ADF 
     *----line cell and rasterize it using an active ordered edge list algorithm.
     */
    ADF_I32   divStatus;
    ADF_I32   j, jMin, jMax;
    ADF_I32   w = distBuf->w;
    ADF_I32   h = distBuf->h;
    ADF_I1616 sampleSpacingX = (distBuf->sampleSpacingX);
    ADF_I1616 sampleSpacingY = (distBuf->sampleSpacingY);
    ADF_I1616 invSampleSpacingX = (distBuf->invSampleSpacingX);
    ADF_I1616 invSampleSpacingY = (distBuf->invSampleSpacingY);
    ADF_I1616 *dist = distBuf->base;
    ADF_I1616 nx = (cell->nx);
    ADF_I1616 ny = (cell->ny);
    ADF_I1616 xs = (cell->x0);
    ADF_I1616 ys = (cell->y0);
    ADF_I1616 xe = (cell->x1);
    ADF_I1616 ye = (cell->y1);
    ADF_I1616 pre_nx = cell->pre_nx;
    ADF_I1616 pre_ny = cell->pre_ny;
    ADF_I1616 next_nx = cell->next_nx;
    ADF_I1616 next_ny = cell->next_ny;

    /**
     *----Edge lists used when rasterizing the implicit ADF line cell
     */
    LineEdge left[3], right[3]; 
    LineEdge *pLeft, *pRight;
    ADF_U32 num_lefts, num_rights; 

    /**
     *----Set constants dx and dy which are required to determine the x and
     *----y-coordinates of the four vertices (x[4] and y[4]) of the implicit ADF 
     *----line cell. The indices into x[] and y[] of the four vertices are stored
     *----in idx[0..3] in clockwise order with idx[0] specifying the index of the
     *----bottommost vertex of the implicit ADF line cell. yMax and yMin store
     *----the y-coordinates of the topmost and bottommost vertices, respectively.
     */ 
    ADF_U32 idx[5];
    ADF_I1616 x[4], y[4];


    ADF_I1616 dx; 
    ADF_I1616 dy;
    ADF_I1616 yMax, yMin;


    /**
     *----The distance increment ddx is the change in the distance d with a unit
     *----increase in x inside the implicit ADF line cell. ddx is used to
     *----increment d when the rasterizer advances to the next sample inside the
     *----implicit ADF line cell when proceeding from left to right. dxyPara and
     *----dxyPerp store the changes in x for a unit increase in y for edges
     *----parallel and perpendicular (in ADF_I1616 fixed point image coordinates)
     *----to the line segment, respectively.
     */   
    ADF_I1616 dxyPara; 
    ADF_I1616 dxy_pre, dxy_next;
    ADF_I1616 bi_pre_tx, bi_pre_ty;
    ADF_I1616 bi_next_tx, bi_next_ty;
    ADF_I1616 tx,ty;
    ADF_I1616 cos_pre, cos_next;


    /**
      *----Determine the four vertices of the implicit ADF line cell. Each vertex
      *----is offset from one of the line segment endpoints by a distance equal to
      *----the filter radius. Scale the vertex positions from ADF_I1616 fixed
      *----point image coordinates to distance buffer coordinates where the
      *----spacing between rows is equal to the spacing between distance samples
      *----in y and the spacing between columns is equal to the spacing between
      *----distance samples in x. If the spacing is non-uniform in x and y (e.g.,
      *----when performing LCD rendering), the rectangular implicit ADF line cell
      *----becomes a parallelogram in distance buffer coordinates. In this case,
      *----edges of the implicit ADF line cell that are parallel to the line
      *----segment in ADF_I1616 fixed point image coordinates remain parallel to
      *----the line segment in distance buffer coordinates, while edges of the
      *----implicit ADF line cell that are perpendicular to the line segment in
      *----ADF_I1616 fixed point image coordinates do not remain perpendicular to
      *----the line segment in distance buffer coordinates.
      */
    ADF_I1616 ddx;

    if ( (invSampleSpacingX == I1616_CONST_1))
        ddx = nx; /* if invSampleSpacingX is I1616_CONST_1, sampleSpacingX is I1616_CONST_1 too */
    else
        ddx = I1616_MUL(nx, sampleSpacingX);

    tx = nx;
    ty = ny;

    if (start == 1)
    {
        bi_pre_tx = (nx + pre_nx) >> 1;
        bi_pre_ty = (ny + pre_ny) >> 1;

        cos_pre = I1616_MUL(bi_pre_tx, tx) + I1616_MUL(bi_pre_ty, ty);
        cos_pre = ABS(cos_pre);

        if (cos_pre > COS_CHECK) 
        {
            ADF_I1616 d = I1616_DIV(filterRad, cos_pre,&divStatus);
            dx = I1616_MUL(bi_pre_tx, d);
            dy = I1616_MUL(bi_pre_ty, d);
        }
        else
        {
            bi_pre_tx = nx;
            bi_pre_ty = ny;
            dx = I1616_MUL(nx,filterRad);
            dy = I1616_MUL(ny,filterRad);
        }

        if ( (invSampleSpacingX == I1616_CONST_1))
        {
            x[0] = xs + dx;
            x[1] = xs - dx; 
        }
        else
        {
            x[0] = I1616_MUL(xs + dx, invSampleSpacingX);
            x[1] = I1616_MUL(xs - dx, invSampleSpacingX); 
        }
        if ( (invSampleSpacingY == I1616_CONST_1) )
        {    
            y[0] = ys + dy;
            y[1] = ys - dy;
        }
        else
        {
            y[0] = I1616_MUL(ys + dy, invSampleSpacingY);
            y[1] = I1616_MUL(ys - dy, invSampleSpacingY);
        }
    }
    else
    {
        cos_pre = *p_cos_pre; 
        if (cos_pre <= COS_CHECK) 
        {
            bi_pre_tx = nx;
            bi_pre_ty = ny;
            dx = I1616_MUL(nx,filterRad);
            dy = I1616_MUL(ny,filterRad);

            if ( (invSampleSpacingX == I1616_CONST_1))
            {      
                x[0] = xs + dx;
                x[1] = xs - dx; 
            }
            else
            {
                x[0] = I1616_MUL(xs + dx, invSampleSpacingX);
                x[1] = I1616_MUL(xs - dx, invSampleSpacingX); 
            }
            if ( (invSampleSpacingY == I1616_CONST_1) )
            {    
                y[0] = ys + dy;
                y[1] = ys - dy;
            }
            else
            {
                y[0] = I1616_MUL(ys + dy, invSampleSpacingY);
                y[1] = I1616_MUL(ys - dy, invSampleSpacingY);
            }
        }
        else
        {
            x[0] = p_xyt[0];
            x[1] = p_xyt[1];
            y[0] = p_xyt[2];
            y[1] = p_xyt[3];
            bi_pre_tx = p_xyt[4];
            bi_pre_ty = p_xyt[5];
        }
    }

    bi_next_tx = (nx + next_nx) >> 1;
    bi_next_ty = (ny + next_ny) >> 1;

    cos_next = I1616_MUL(bi_next_tx, tx) + I1616_MUL(bi_next_ty, ty);    
    cos_next = ABS(cos_next);

    if (cos_next > COS_CHECK) 
    {
        ADF_I1616 d = I1616_DIV(filterRad, cos_next,&divStatus);
        dx = I1616_MUL(bi_next_tx, d);
        dy = I1616_MUL(bi_next_ty, d);
    }
    else
    {
        /* make it vertical */
        bi_next_tx = nx;    
        bi_next_ty = ny;    
        dx = I1616_MUL(bi_next_tx,filterRad);
        dy = I1616_MUL(bi_next_ty,filterRad);
    }

    if ( (invSampleSpacingX == I1616_CONST_1))
    {      
        x[2] = xe - dx; 
        x[3] = xe + dx; 
    }
    else
    {
        x[2] = I1616_MUL(xe - dx, invSampleSpacingX); 
        x[3] = I1616_MUL(xe + dx, invSampleSpacingX);
    }
    if ( (invSampleSpacingY == I1616_CONST_1) )
    {    
        y[2] = ye - dy;
        y[3] = ye + dy;
    }
    else
    {
        y[2] = I1616_MUL(ye - dy, invSampleSpacingY);
        y[3] = I1616_MUL(ye + dy, invSampleSpacingY);
    }

    *p_cos_pre = cos_next;
    p_xyt[0] = x[3];
    p_xyt[1] = x[2];;
    p_xyt[2] = y[3];
    p_xyt[3] = y[2];
    p_xyt[4] = bi_next_tx;
    p_xyt[5] = bi_next_ty;

    /* process horizontal line */
    if (cell->y0 == cell->y1)
    {
        if ((start == 1))
            dxy_pre = I1616_DIV(bi_pre_tx, bi_pre_ty, &divStatus);
        else
        {
            if (cos_pre <= COS_CHECK)
                dxy_pre = I1616_DIV(bi_pre_tx, bi_pre_ty, &divStatus);
            else
                dxy_pre = p_xyt[6];
        }

        dxy_next = I1616_DIV(bi_next_tx, bi_next_ty, &divStatus);
        p_xyt[6] = dxy_next;

        ProcessHorizontalLineCell_trap(cell, distBuf, filterRad, wnb, bcell,
                                       dxy_pre,dxy_next,x,y);
        return;
    }

    /**
    *----Determine the change in x for a unit increase in y in distance buffer
    *----coordinates for edges parallel and perpendicular (in ADF_I1616 fixed
    *----point image coordinates) to the line segment. Since this line segment
    *----is diagonal (and therefore not horizontal), y[3] and y[1] must be
    *----different from y[0]. Consequently, ye-ys is never zero, so divisions by
    *----zero cannot occur. FixedMathNote: the following two divisions may
    *----overflow the ADF_I1616 fixed point representation. Overflow will only
    *----occur if abs(x[3] - x[0]) is significantly greater than abs(dy) or if
    *----abs(x[1] - x[0]) is significantly greater than abs(dy). This situation
    *----occurs when the diagonal line segment is nearly vertical or nearly
    *----horizontal. Consequently, if the line segment is nearly vertical (i.e.,
    *----abs(nx) > abs(ny)), process this line segment as a vertical line
    *----segment. If the line segment is nearly horizontal, process this line
    *----segment as a horizontal line segment.
    */

    dxyPara = I1616_DIV(xe-xs, ye-ys, &divStatus); 

    if ((start == 1))
        dxy_pre = I1616_DIV(bi_pre_tx, bi_pre_ty, &divStatus);
    else
    {
        if (cos_pre <= COS_CHECK)
            dxy_pre = I1616_DIV(bi_pre_tx, bi_pre_ty, &divStatus);
        else
            dxy_pre = p_xyt[6];
    }

    dxy_next = I1616_DIV(bi_next_tx, bi_next_ty,&divStatus);
    p_xyt[6] = dxy_next;


    /**
     *----Set the left and right edge properties. There at most 3 left edges
     *----and 3 right edges for a total of at most 4 edges. Each edge has a 
     *----top scan line, a starting x point, a starting distance value, and
     *----a slope dx/dy, and a slope in distance (dd/dy).
     */
    Set_Left_Right_Edges(cell, x, y, idx, dxyPara, dxy_pre, dxy_next,
                         bi_pre_tx, bi_pre_ty, bi_next_tx, bi_next_ty,
                         cos_pre, cos_next, invSampleSpacingX, sampleSpacingY,
                         invSampleSpacingY,
                         filterRad, left, right, &num_lefts, &num_rights);


    /**
     *----Determine the topmost (i.e., the last) scanline (i.e., jMax) of the
     *----implicit ADF line cell to be rasterized (note the exclusion of topmost
     *----vertices falling exactly on a scanline). The cell is not rasterized if
     *----it lies below the bottom of the distance buffer.
     */
    yMax = y[idx[4]];
    if (yMax < 0) return;
    jMax = I1616_TO_I32(yMax);
    if (jMax >= h) jMax = h - 1;
    else if (yMax == I32_TO_I1616(jMax)) jMax -= 1;


    /**
     *----Determine the bottommost (i.e., the first) scanline (i.e., jMin) of the
     *----implicit ADF line cell to be rasterized (note the inclusion of
     *----bottommost vertices falling exactly on a scanline).
     */
    yMin = y[idx[0]];
    jMin = I1616_TO_I32(yMin);
    if (yMin < 0) jMin = 0;
    else if (yMin != I32_TO_I1616(jMin)) jMin += 1;


    /**
     *----Set a pointer to the first left edge. Skip the first left edge if it is
     *----either 1) nearly horizontal and does not cross the first scanline, or
     *----2) below the bottom of the distance buffer.
     */
    if (left[0].yTop > I32_TO_I1616(jMin)) 
    {
        /**
         *----Set the pointer to the first left edge
         */
        pLeft = &(left[0]);
    } 
    else 
    {
        if (num_lefts > 1)
        {
            if (left[1].yTop > I32_TO_I1616(jMin))
                pLeft = &(left[1]);
            else
            {
                if (num_lefts == 3 )
                {
                    if (left[2].yTop > I32_TO_I1616(jMin))
                        pLeft = &(left[2]);
                    else
                        return;
                }
                else
                    return;
            }
        }
        else
            return;
    }

    /**
     *----Set a pointer to the first right edge. Skip the first right edge if it
     *----is either 1) nearly horizontal and does not cross the first scanline,
     *----or 2) below the bottom of the distance buffer.
     */
    if (right[0].yTop > I32_TO_I1616(jMin)) {

        /**
         *----Set the pointer to the first right edge
         */
        pRight = &(right[0]);
    } 
    else 
    {
        if (num_rights > 1)
        {
            if (right[1].yTop > I32_TO_I1616(jMin))
                pRight = &(right[1]);
            else
            {
                if (num_rights == 3 )
                {
                    if (right[2].yTop > I32_TO_I1616(jMin))
                        pRight = &(right[2]);
                    else
                        return;
                }
                else
                    return;
            }
        }
        else
            return;
    }


    if( wnb )
    {
        ADF_I32 t = MAX(w,h);

        if(filterRad > I1616_CONST_1)
            t = MIN((t + (t>>5)*1024),4096);

        /**
        *----This is a closed loop contour path for which a winding number buffer
        *----exists. 
        *----Rasterize the implicit ADF line cell from bottom to top (i.e., jMin to
        *----jMax), left to right (i.e., i to iMax). This assumes ink-on-the-right.
        *----For each interior sample point, determine the distance from the line
        *----segment to the sample point. Combine distances in a manner that depends
        *----on whether the sample point is inside the contour or outside the
        *----contour as determined by the winding number buffer.
        */
        j = jMin;
        while (j <= jMax) {
            ADF_I32      i, iMax;
            ADF_I1616 d;
            ADF_I1616 *pDist;
            ADF_I1616 *pBcell=0;
            ADF_I8 *pWinding;
            ADF_I32 sample;

            /**
            *----Determine the column index i of the distance buffer of the first
            *----sample point that is on or to the right of the current left
            *----edge. If the current left edge intersects the current scanline at a
            *----point outside and to the left of the distance buffer, set the
            *----column index i to zero (i.e., the leftmost column of the distance
            *----buffer).
            */
            i = I1616_TO_I32(pLeft->x);
            if (pLeft->x < 0) i = 0;
            else if (pLeft->x != I32_TO_I1616(i)) i += 1;


            /**
            *----Determine the column index iMax of the distance buffer of the last
            *----sample point that is to the left of the current right edge. If the
            *----current right edge intersects the current scanline at a point
            *----outside and to the right of the distance buffer, set the column
            *----index iMax to the rightmost column of the distance buffer.
            */
            iMax = I1616_TO_I32(pRight->x);
            if (iMax >= w) iMax = w - 1;
            else if (pRight->x == I32_TO_I1616(iMax)) iMax -= 1;


            /**
            *----Determine the distance d from the line segment represented by the
            *----implicit ADF line cell to the sample point i of the current
            *----scanline
            */
            d = pLeft->d + I1616_MUL(I32_TO_I1616(i) - pLeft->x, ddx);

            /**
            *----Set a pointer to the entry in the distance buffer corresponding to
            *----the first sample point i of the current scanline
            */
            sample = j * w + i;
            pDist = &dist[sample];
            if(bcell)
                pBcell = &bcell[sample];
            pWinding = &wnb[sample];

            /**
            *----Increment the signed distance d across the current scanline to
            *----determine the distance from the line segment to each interior
            *----sample point of the current scanline. Combine each distance with
            *----its corresponding distance in the distance buffer.
            */ 
            while (i <= iMax) {   
                /**
                *----Combine the current line distance with its corresponding distance 
                *----in the distance buffer
                */
                if (*pWinding == 1)
                {
                    if ((d >= -t) && (d < *pDist))
                        *pDist = d;  /* inside pixel */
                }
                else if (*pWinding == 0)
                {
                    if ((d <= t) && (d > *pDist))
                        *pDist = d;  /* outside pixel */
                }
                else /* overlapping contours */
                {
                    if (pBcell && d > 0)
                    {
                        /**
                        *----Overlapping boundaries.
                        */
                        if (*pBcell == 0)
                        {
                            /**
                            *----This is the first visit to this pixel and the
                            *----winding number is greater than one. We may visit
                            *----this pixel again, so store the current distance
                            *----in the boundary cell buffer in case we do, but do not set
                            *----the distance buffer because this may be an interior
                            *----pixel with only one visit.
                            */
                            *pBcell = d;
                        }
                        else if (*pBcell > 0)
                        {
                            /**
                            *----A previous overlapping boundary cell point was set.
                            *----This is the second or higher visit to this pixel.
                            *----We have three choices for setting the distance.
                            *----These are (d,*pBcell,*pDist). We want to take
                            *----the distance in the middle and keep the smaller
                            *----value in bcell in case there are subsequent visits
                            *----Ultimately, this process picks the larger of the
                            *----two smallest distances.
                            *----Note: on the second visit to this pixel,
                            *----*pDist==LARGE_INSIDE_DISTANCE so *pDist will always
                            *----be set to the maximum of (d,*pBcell) and *pBcell will
                            *----be set to the minimum of (d,*pBcell).
                            */
                            if(d > *pBcell)
                            {
                                if(d < *pDist)
                                    *pDist = d;
                            }
                            else
                            {
                                *pDist = *pBcell;
                                *pBcell = d;
                            }
                        }
                    } /* if (bcell) */
                }
                /**
                *----Update the index of the current sample point, the
                *----interpolated distance value, and the pointer to the distance
                *----buffer corresponding to the current sample point
                */
                i++; 
                d += ddx;
                pDist++;
                pWinding++;
                if (pBcell)
                    pBcell++;
            }


            /**
            *----Update the index of the current scanline and the left and right
            *----edge data
            */
            j++;
            if (I32_TO_I1616(j) >= pLeft->yTop) 
            {
                while(I32_TO_I1616(j) >= pLeft->yTop)
                {
                    pLeft++;
                    if (pLeft > &left[num_lefts - 1])
                        return;
                }
            }
            else 
            {
                pLeft->x += pLeft->dxy; 
                pLeft->d += pLeft->ddy;
            }
            if (I32_TO_I1616(j) >= pRight->yTop) 
            {
                while(I32_TO_I1616(j) >= pRight->yTop)
                {
                    pRight++;
                    if (pRight > &right[num_rights - 1])
                        return;
                }
            }
            else 
                pRight->x += pRight->dxy;
        } /* while (j <= jMax) */
    } /* if ( wnb ) */
}

/**
 *------------------process line cell for rectangular line cells --------------------
 *-----------------------------------------------------------------------------------
 */
static ADF_Void ProcessLineCell_rect(CellData *cell, DistBuffer *distBuf, 
                                     ADF_I1616 filterRad, ADF_I8 *wnb, 
                                     ADF_I1616 *bcell)
{
    /**
     *----Determine if the line segment is vertical, horizontal, or diagonal (i.e.,
     *----neither vertical nor horizontal)
     */
    if (cell->x0 == cell->x1) {


        /**
         *----This line segment is vertical. Process the line segment using
         *----ProcessVerticalLineCell(), whose implementation is optimized for
         *----vertical line segments.
         */
        ProcessVerticalLineCell:
        ProcessVerticalLineCell_rect(cell, distBuf, filterRad, wnb, bcell);


    } else if (cell->y0 == cell->y1) {


        /**
         *----This line segment is horizontal. Process the line segment using
         *----ProcessHorizontalLineCell(), whose implementation is optimized for
         *----horizontal line segments.
         */
        ProcessHorizontalLineCell:
        ProcessHorizontalLineCell_rect(cell, distBuf, filterRad, wnb, bcell);



    } else {


        /**
         *----This line segment is diagonal. Construct an implicit ADF line cell and
         *----rasterize it using an active ordered edge list algorithm.
         */
        ADF_I32   divStatus;
        ADF_I32   j, jMin, jMax;
        ADF_I32   w = distBuf->w;
        ADF_I32   h = distBuf->h;
        ADF_I1616 sampleSpacingX = (distBuf->sampleSpacingX);
        ADF_I1616 sampleSpacingY = (distBuf->sampleSpacingY);
        ADF_I1616 invSampleSpacingX = (distBuf->invSampleSpacingX);
        ADF_I1616 invSampleSpacingY = (distBuf->invSampleSpacingY);
        ADF_I1616 *dist = distBuf->base, *pBcell = 0;
        ADF_I1616 nx = (cell->nx);
        ADF_I1616 ny = (cell->ny);
        ADF_I1616 xs = (cell->x0);
        ADF_I1616 ys = (cell->y0);
        ADF_I1616 xe = (cell->x1);
        ADF_I1616 ye = (cell->y1);


        /**
         *----Edge lists used when rasterizing the implicit ADF line cell
         */
        LineEdge left[2], right[2];
        LineEdge *pLeft, *pRight;


        /**
         *----Set constants dx and dy which are required to determine the x and
         *----y-coordinates of the four vertices (x[4] and y[4]) of the implicit ADF
         *----line cell. The indices into x[] and y[] of the four vertices are stored
         *----in idx[0..3] in clockwise order with idx[0] specifying the index of the
         *----bottommost vertex of the implicit ADF line cell. yMax and yMin store
         *----the y-coordinates of the topmost and bottommost vertices, respectively.
         */
        ADF_U32   idx[4];
        ADF_I1616 x[4], y[4];
        ADF_I1616 dx = I1616_MUL(nx, filterRad);
        ADF_I1616 dy = I1616_MUL(ny, filterRad);
        ADF_I1616 yMax, yMin;


        /**
         *----Constants required to initialize edge data for the implicit ADF line
         *----cell
         */
        ADF_I1616 dy0, dy1;
        ADF_I1616 d0, d1, dxy0, dxy1, ddy0, ddy1;


        /**
         *----The distance increment ddx is the change in the distance d with a unit
         *----increase in x inside the implicit ADF line cell. ddx is used to
         *----increment d when the rasterizer advances to the next sample inside the
         *----implicit ADF line cell when proceeding from left to right. dxyPara and
         *----dxyPerp store the changes in x for a unit increase in y for edges
         *----parallel and perpendicular (in ADF_I1616 fixed point image coordinates)
         *----to the line segment, respectively.
         */
        ADF_I1616 dxyPara;
        ADF_I1616 dxyPerp;


        /**
         *----Determine the four vertices of the implicit ADF line cell. Each vertex
         *----is offset from one of the line segment endpoints by a distance equal to
         *----the filter radius. Scale the vertex positions from ADF_I1616 fixed
         *----point image coordinates to distance buffer coordinates where the
         *----spacing between rows is equal to the spacing between distance samples
         *----in y and the spacing between columns is equal to the spacing between
         *----distance samples in x. If the spacing is non-uniform in x and y (e.g.,
         *----when performing LCD rendering), the rectangular implicit ADF line cell
         *----becomes a parallelogram in distance buffer coordinates. In this case,
         *----edges of the implicit ADF line cell that are parallel to the line
         *----segment in ADF_I1616 fixed point image coordinates remain parallel to
         *----the line segment in distance buffer coordinates, while edges of the
         *----implicit ADF line cell that are perpendicular to the line segment in
         *----ADF_I1616 fixed point image coordinates do not remain perpendicular to
         *----the line segment in distance buffer coordinates.
         */

        ADF_I1616 ddx;

        if ( (invSampleSpacingX == I1616_CONST_1))
        {
            ddx = nx; /* if invSampleSpacingX is I1616_CONST_1, sampleSpacingX is I1616_CONST_1 too */
            x[0] = xs + dx;
            x[1] = xs - dx; 
            x[2] = xe - dx; 
            x[3] = xe + dx;
        }
        else
        {
            ddx = I1616_MUL(nx, sampleSpacingX); 
            x[0] = I1616_MUL(xs + dx, invSampleSpacingX);
            x[1] = I1616_MUL(xs - dx, invSampleSpacingX); 
            x[2] = I1616_MUL(xe - dx, invSampleSpacingX); 
            x[3] = I1616_MUL(xe + dx, invSampleSpacingX);
        }
        
        if ( (invSampleSpacingY == I1616_CONST_1) )
        {
            y[0] = ys + dy;
            y[1] = ys - dy;
            y[2] = ye - dy;
            y[3] = ye + dy;
        }
        else
        {
            y[0] = I1616_MUL(ys + dy, invSampleSpacingY);
            y[1] = I1616_MUL(ys - dy, invSampleSpacingY);
            y[2] = I1616_MUL(ye - dy, invSampleSpacingY);
            y[3] = I1616_MUL(ye + dy, invSampleSpacingY);
        }


        /*lint -e801  Info -- Use of goto is deprecated */

        /**
         *----Determine the change in x for a unit increase in y in distance buffer
         *----coordinates for edges parallel and perpendicular (in ADF_I1616 fixed
         *----point image coordinates) to the line segment. Since this line segment
         *----is diagonal (and therefore not horizontal), y[3] and y[1] must be
         *----different from y[0]. Consequently, dy is never zero, so divisions by
         *----zero cannot occur. FixedMathNote: the following two divisions may
         *----overflow the ADF_I1616 fixed point representation. Overflow will only
         *----occur if abs(x[3] - x[0]) is significantly greater than abs(dy) or if
         *----abs(x[1] - x[0]) is significantly greater than abs(dy). This situation
         *----occurs when the diagonal line segment is nearly vertical or nearly
         *----horizontal. Consequently, if the line segment is nearly vertical (i.e.,
         *----abs(nx) > abs(ny)), process this line segment as a vertical line
         *----segment. If the line segment is nearly horizontal, process this line
         *----segment as a horizontal line segment.
         */
        dy = y[3] - y[0];
        dxyPara = I1616_DIV(x[3] - x[0], dy, &divStatus);
        if (divStatus != ADF_FIXED_MATH_NO_ERROR) {
            if (I1616_ABS(nx) > I1616_ABS(ny)) goto ProcessVerticalLineCell;
            else goto ProcessHorizontalLineCell;
        }
        dy = y[1] - y[0];
        dxyPerp = I1616_DIV(x[1] - x[0], dy, &divStatus);
        if (divStatus != ADF_FIXED_MATH_NO_ERROR) {
            if (I1616_ABS(nx) > I1616_ABS(ny)) goto ProcessVerticalLineCell;
            else goto ProcessHorizontalLineCell;
        }


        /**
         *----Order the four vertices of the implicit ADF line cell and determine the
         *----data required to initialize cell edges for rasterizing. The vertices
         *----are ordered in a clockwise direction, starting from the bottommost
         *----vertex.
         */
        if (ys <= ye) {
            if (xs <= xe) {


                /**
                 *----This line segment runs from bottom to top and left to right.
                 *----Set the vertex indices to order the cell vertices, starting
                 *----from the bottommost vertex and continuing in a clockwise
                 *----direction.
                 */
                idx[0] = 0; idx[1] = 1; idx[2] = 2; idx[3] = 3;


                /**
                 *----For each edge, the change dxy in x-intercept with a unit
                 *----increase in y depends on whether the edge is parallel or
                 *----perpendicular (in ADF_I1616 fixed point image coordinates) to
                 *----the line segment. Set these dxy values for the first (i.e.,
                 *----bottommost left edge) and second left edges (i.e., set dxy0 and
                 *----dxy1, respectively). Because the implicit ADF line cell is a
                 *----parallelogram in distance buffer coordinates, dxy0 and dxy1
                 *----also serve as the dxy for the second (i.e., topmost right edge)
                 *----and first right edges, respectively.
                 */
                dxy0 = dxyPerp;
                dxy1 = dxyPara;


                /**
                 *----Determine the distance value d0 at the first vertex of the
                 *----first left edge and the distance value d1 at the first vertex
                 *----of the second left edge. Compute the change in the distance
                 *----value along the first left edge (ddy0) and the change in the
                 *----distance value along the second left edge (ddy1) when advancing
                 *----from scanline to scanline while rasterizing the distance buffer
                 *----from bottom to top.
                 */
                d0 =  filterRad;
                d1 =  I1616_NEGATE(filterRad);
                ddy0 = I1616_DIV(sampleSpacingY, cell->ny, &divStatus);
                ddy1 = 0;


                /**
                 *----FixedMathNote: the division above may overflow the ADF_I1616
                 *----fixed point representation. Since sampleSpacingY is a constant
                 *----whose mathematical value is either 1, 1/3, or 1/n overflow can 
                 *----only occur if cell->ny is very small. Note that if cell->ny 
                 *----(i.e., the y-coordinate of the normal vector to the line segment) is very
                 *----small, then the line segment must be nearly vertical. Process
                 *----this line segment as a vertical line segment if overflow
                 *----occurs.
                 */
                if (divStatus != ADF_FIXED_MATH_NO_ERROR)
                    goto ProcessVerticalLineCell;


            } else {


                /**
                 *----This line segment runs from bottom to top and right to left.
                 *----Set the vertex indices to order the cell vertices, starting
                 *----from the bottommost vertex and continuing in a clockwise
                 *----direction.
                 */
                idx[0] = 1; idx[1] = 2; idx[2] = 3; idx[3] = 0;


                /**
                 *----For each edge, the change dxy in x-intercept with a unit
                 *----increase in y depends on whether the edge is parallel or
                 *----perpendicular (in ADF_I1616 fixed point image coordinates) to
                 *----the line segment. Set these dxy values for the first (i.e.,
                 *----bottommost left edge) and second left edges (i.e., set dxy0 and
                 *----dxy1, respectively). Because the implicit ADF line cell is a
                 *----parallelogram in distance buffer coordinates, dxy0 and dxy1
                 *----also serve as the dxy for the second (i.e., topmost right edge)
                 *----and first right edges, respectively.
                 */
                dxy0 = dxyPara;
                dxy1 = dxyPerp;


                /**
                 *----Determine the distance value d0 at the first vertex of the
                 *----first left edge and the distance value d1 at the first vertex
                 *----of the second left edge. Compute the change in the distance
                 *----value along the first left edge (ddy0) and the change in the
                 *----distance value along the second left edge (ddy1) when advancing
                 *----from scanline to scanline while rasterizing the distance buffer
                 *----from bottom to top.
                 */
                d0 = I1616_NEGATE(filterRad);
                d1 = I1616_NEGATE(filterRad);
                ddy0 = 0;
                ddy1 = I1616_DIV(sampleSpacingY, cell->ny, &divStatus);


                /**
                 *----FixedMathNote: the division above may overflow the ADF_I1616
                 *----fixed point representation. Since sampleSpacingY is a constant
                 *----whose mathematical value is either 1, 1/3, or 1/n overflow can only
                 *----occur if cell->ny is very small. Note that if cell->ny (i.e., the
                 *----y-coordinate of the normal vector to the line segment) is very
                 *----small, then the line segment must be nearly vertical. Process
                 *----this line segment as a vertical line segment if overflow
                 *----occurs.
                 */
                if (divStatus != ADF_FIXED_MATH_NO_ERROR)
                    goto ProcessVerticalLineCell;
            }


        } else {
            if (xs <= xe) {


                /**
                 *----This line segment runs from top to bottom and left to right.
                 *----Set the vertex indices to order the cell vertices, starting
                 *----from the bottommost vertex and continuing in a clockwise
                 *----direction.
                 */
                idx[0] = 3; idx[1] = 0; idx[2] = 1; idx[3] = 2;


                /**
                 *----For each edge, the change dxy in x-intercept with a unit
                 *----increase in y depends on whether the edge is parallel or
                 *----perpendicular (in ADF_I1616 fixed point image coordinates) to
                 *----the line segment. Set these dxy values for the first (i.e.,
                 *----bottommost left edge) and second left edges (i.e., set dxy0 and
                 *----dxy1, respectively). Because the implicit ADF line cell is a
                 *----parallelogram in distance buffer coordinates, dxy0 and dxy1
                 *----also serve as the dxy for the second (i.e., topmost right edge)
                 *----and first right edges, respectively.
                 */
                dxy0 = dxyPara; 
                dxy1 = dxyPerp;


                /**
                 *----Determine the distance value d0 at the first vertex of the
                 *----first left edge and the distance value d1 at the first vertex
                 *----of the second left edge. Compute the change in the distance
                 *----value along the first left edge (ddy0) and the change in the
                 *----distance value along the second left edge (ddy1) when advancing
                 *----from scanline to scanline while rasterizing the distance buffer
                 *----from bottom to top.
                 */
                d0 = filterRad;
                d1 = filterRad;
                ddy0 = 0;
                ddy1 = I1616_DIV(sampleSpacingY, cell->ny, &divStatus);

                
                /**
                 *----FixedMathNote: the division above may overflow the ADF_I1616
                 *----fixed point representation. Since sampleSpacingY is a constant
                 *----whose mathematical value is either 1, 1/3, or 1/n, overflow can 
                 *----only occur if cell->ny is very small. Note that if cell->ny (i.e., the
                 *----y-coordinate of the normal vector to the line segment) is very
                 *----small, then the line segment must be nearly vertical. Process
                 *----this line segment as a vertical line segment if overflow
                 *----occurs.
                 */
                if (divStatus != ADF_FIXED_MATH_NO_ERROR)
                    goto ProcessVerticalLineCell;


            } else {


                /**
                 *----This line segment runs from top to bottom and right to
                 *----left. Set the vertex indices to order the cell vertices,
                 *----starting from the bottommost vertex and continuing in a
                 *----clockwise direction.
                 */
                idx[0] = 2; idx[1] = 3; idx[2] = 0; idx[3] = 1;


                /**
                 *----For each edge, the change dxy in x-intercept with a unit
                 *----increase in y depends on whether the edge is parallel or
                 *----perpendicular (in ADF_I1616 fixed point image coordinates) to
                 *----the line segment. Set these dxy values for the first (i.e.,
                 *----bottommost left edge) and second left edges (i.e., set dxy0 and
                 *----dxy1, respectively). Because the implicit ADF line cell is a
                 *----parallelogram in distance buffer coordinates, dxy0 and dxy1
                 *----also serve as the dxy for the second (i.e., topmost right edge)
                 *----and first right edges, respectively.
                 */
                dxy0 = dxyPerp; 
                dxy1 = dxyPara;


                /**
                 *----Determine the distance value d0 at the first vertex of the
                 *----first left edge and the distance value d1 at the first vertex
                 *----of the second left edge. Compute the change in the distance
                 *----value along the first left edge (ddy0) and the change in the
                 *----distance value along the second left edge (ddy1) when advancing
                 *----from scanline to scanline while rasterizing the distance buffer
                 *----from bottom to top.
                 */
                d0 = I1616_NEGATE(filterRad);
                d1 =  filterRad;
                ddy0 = I1616_DIV(sampleSpacingY, cell->ny, &divStatus);
                ddy1 = 0;


                /**
                 *----FixedMathNote: the division above may overflow the ADF_I1616
                 *----fixed point representation. Since sampleSpacingY is a constant
                 *----whose mathematical value is either 1, 1/3, or 1/n, overflow can 
                 *----only occur if cell->ny is very small. Note that if cell->ny (i.e., the
                 *----y-coordinate of the normal vector to the line segment) is very
                 *----small, then the line segment must be nearly vertical. Process
                 *----this line segment as a vertical line segment if overflow
                 *----occurs.
                 */
                if (divStatus != ADF_FIXED_MATH_NO_ERROR)
                    goto ProcessVerticalLineCell;
            }
        }

        /*lint +e801  Info -- Use of goto is deprecated */

        /**
         *----Determine the topmost (i.e., the last) scanline (i.e., jMax) of the
         *----implicit ADF line cell to be rasterized (note the exclusion of topmost
         *----vertices falling exactly on a scanline). The cell is not rasterized if
         *----it lies below the bottom of the distance buffer.
         */
        yMax = y[idx[2]];
        if (yMax < 0) return;
        jMax = I1616_TO_I32(yMax);
        if (jMax >= h) jMax = h - 1;
        else if (yMax == I32_TO_I1616(jMax)) jMax -= 1;


        /**
         *----Determine the bottommost (i.e., the first) scanline (i.e., jMin) of the
         *----implicit ADF line cell to be rasterized (note the inclusion of
         *----bottommost vertices falling exactly on a scanline).
         */
        yMin = y[idx[0]];
        jMin = I1616_TO_I32(yMin);
        if (yMin < 0) jMin = 0;
        else if (yMin != I32_TO_I1616(jMin)) jMin += 1;


        /**
         *----Initialize the first left edge. dy0 is the height of jMin (i.e., the
         *----first scanline to be rasterized) above the bottommost vertex of the
         *----implicit ADF line cell; dy0 is used to determine the first x-intercept
         *----and the distance at the first x-intercept for the first left
         *----edge. Since this height is the same for the first right edge, dy0 is
         *----reused below when initializing the first right edge.
         */
        dy0 = I32_TO_I1616(jMin) - y[idx[0]];
        left[0].yTop = y[idx[1]];
        left[0].dxy = dxy0;
        left[0].ddy = ddy0;
        left[0].x = x[idx[0]] + I1616_MUL(dy0, dxy0);
        left[0].d = d0 + I1616_MUL(dy0, ddy0);


        /**
         *----Initialize the second left edge. dy1 is the height of the first
         *----scanline intersecting the second left edge above the bottommost vertex
         *----of the second left edge; dy1 is used to determine the first x-intercept
         *----and the distance at the first x-intercept for the second left edge.
         */
        if (y[idx[1]] <= 0) dy1 = I1616_NEGATE(y[idx[1]]);
        else {
            dy1 = I32_TO_I1616(I1616_TO_I32(y[idx[1]])) - y[idx[1]];
            if (dy1 != 0) dy1 = dy1 + I1616_CONST_1;
        }
        left[1].yTop = yMax;
        left[1].dxy = dxy1;
        left[1].ddy = ddy1;
        left[1].x = x[idx[1]] + I1616_MUL(dy1, dxy1);
        left[1].d = d1 +I1616_MUL(dy1, ddy1);


        /**
         *----Initialize the first right edge
         */
        right[0].yTop = y[idx[3]];
        right[0].dxy = dxy1;
        right[0].x = x[idx[0]] + I1616_MUL(dy0, dxy1);


        /**
         *----Initialize the second right edge. dy1 is the height of the first
         *----scanline intersecting the second right edge above the bottommost vertex
         *----of the second right edge; dy1 is used to determine the first
         *----x-intercept and the distance at the first x-intercept for the second
         *----right edge.
         */
        if (y[idx[3]] <= 0) dy1 = I1616_NEGATE(y[idx[3]]);
        else {
            dy1 = I32_TO_I1616(I1616_TO_I32(y[idx[3]])) - y[idx[3]];
            if (dy1 != 0) dy1 = dy1 + I1616_CONST_1;
        }
        right[1].yTop = yMax;
        right[1].dxy = dxy0;
        right[1].x = x[idx[3]] + I1616_MUL(dy1, dxy0);


        /**
         *----Set a pointer to the first left edge. Skip the first left edge if it is
         *----either 1) nearly horizontal and does not cross the first scanline, or
         *----2) below the bottom of the distance buffer.
         */
        if (left[0].yTop > I32_TO_I1616(jMin)) {


            /**
             *----Set the pointer to the first left edge
             */
            pLeft = &(left[0]);


        } else {


            /**
             *----Skip to the second left edge
             */
            pLeft = &(left[1]);
        }


        /**
         *----Set a pointer to the first right edge. Skip the first right edge if it
         *----is either 1) nearly horizontal and does not cross the first scanline,
         *----or 2) below the bottom of the distance buffer.
         */
        if (right[0].yTop > I32_TO_I1616(jMin)) {


            /**
             *----Set the pointer to the first right edge
             */
            pRight = &(right[0]);


        } else {


            /**
             *----Skip to the second right edge
             */
            pRight = &(right[1]);
        }


        if( wnb )
        {
            /**
            *----This is a closed loop contour path for which a winding number buffer
            *----exists. 
            *----Rasterize the implicit ADF line cell from bottom to top (i.e., jMin to
            *----jMax), left to right (i.e., i to iMax). This assumes ink-on-the-right.
            *----For each interior sample point, determine the distance from the line 
            *----segment to the sample point. Combine distances in a manner that depends 
            *----on whether the sample point is inside the contour or outside the 
            *----contour as determined by the winding number buffer. 
            */
            j = jMin;
            while (j <= jMax) {
                ADF_I32      i, iMax;
                ADF_I1616 d;
                ADF_I1616 *pDist;
                ADF_I8 *pWinding;
                ADF_I32 sample;


                /**
                *----Determine the column index i of the distance buffer of the first
                *----sample point that is on or to the right of the current left
                *----edge. If the current left edge intersects the current scanline at a
                *----point outside and to the left of the distance buffer, set the
                *----column index i to zero (i.e., the leftmost column of the distance
                *----buffer).
                */
                i = I1616_TO_I32(pLeft->x);
                if (pLeft->x < 0) i = 0;
                else if (pLeft->x != I32_TO_I1616(i)) i += 1;


                /**
                *----Determine the column index iMax of the distance buffer of the last
                *----sample point that is to the left of the current right edge. If the
                *----current right edge intersects the current scanline at a point
                *----outside and to the right of the distance buffer, set the column
                *----index iMax to the rightmost column of the distance buffer.
                */
                iMax = I1616_TO_I32(pRight->x);
                if (iMax >= w) iMax = w - 1;
                else if (pRight->x == I32_TO_I1616(iMax)) iMax -= 1;


                /**
                *----Determine the distance d from the line segment represented by the
                *----implicit ADF line cell to the sample point i of the current
                *----scanline
                */
                d = pLeft->d + I1616_MUL(I32_TO_I1616(i) - pLeft->x, ddx);


                /**
                *----Set a pointer to the entry in the distance buffer corresponding to
                *----the first sample point i of the current scanline
                */
                sample = j * w + i;
                pDist = &dist[sample];
                pWinding = &wnb[sample];


                /**
                *----Increment the signed distance d across the current scanline to
                *----determine the distance from the line segment to each interior
                *----sample point of the current scanline. Combine each distance with
                *----its corresponding distance in the distance buffer.
                */
                while (i <= iMax) {


                    /**
                    *----Combine the current line distance with its corresponding distance 
                    *----in the distance buffer
                    */
                    if (*pWinding == 1)
                    {
                        if ((d > -1024) && (d < *pDist)) *pDist = d;  /* inside pixel */
                    }
                    else if (*pWinding == 0)
                    {
                        if ((d < 1024) && (d > *pDist)) *pDist = d;  /* outside pixel */
                    }
                    else
                    {
                         if (pBcell && d > 0)
                         {
                             /**
                             *----Overlapping boundaries. 
                             */
                             if (*pBcell == 0)
                             {
                                 /**
                                 *----This is the first visit to this pixel and the
                                 *----winding number is greater than one. We may visit
                                 *----this pixel again, so store the current distance
                                 *----in the boundary cell buffer in case we do, but do not set 
                                 *----the distance buffer because this may be an interior 
                                 *----pixel with only one visit.
                                 */
                                 *pBcell = d;
                             }
                             else if (*pBcell > 0)
                             {
                                 /**
                                 *----A previous overlapping boundary cell point was set.
                                 *----This is the second or higher visit to this pixel. 
                                 *----We have three choices for setting the distance.
                                 *----These are (d,*pBcell,*pDist). We want to take
                                 *----the distance in the middle and keep the smaller
                                 *----value in bcell in case there are subsequent visits.
                                 *----Ultimately, this process picks the larger of the 
                                 *----two smallest distances.
                                 *----Note: on the second visit to this pixel, 
                                 *----*pDist==LARGE_INSIDE_DISTANCE so *pDist will always
                                 *----be set to the maximum of (d,*pBcell) and *pBcell will
                                 *----be set to the minimum of (d,*pBcell).
                                 */
                                 if(d > *pBcell)
                                 {
                                     if(d < *pDist)
                                         *pDist = d;
                                 }
                                 else
                                 {
                                     *pDist = *pBcell;
                                     *pBcell = d;
                                 }
                             }
                         }
                    } /* if (bcell) */


                    /**
                    *----Update the index of the current sample point, the
                    *----interpolated distance value, and the pointer to the distance
                    *----buffer corresponding to the current sample point
                    */
                    i++;
                    d += ddx;
                    pDist++;
                    pWinding++;
                }


                /**
                *----Update the index of the current scanline and the left and right
                *----edge data
                */
                j++;
                if (I32_TO_I1616(j) >= pLeft->yTop) pLeft++;
                else {pLeft->x += pLeft->dxy; pLeft->d += pLeft->ddy;}
                if (I32_TO_I1616(j) >= pRight->yTop) pRight++;
                else pRight->x += pRight->dxy;
            } /* while (j <= jMax) */
        } /* if ( wnb ) */

        else

        {

            /**
            *----This is a uniform stroke path for which a winding number buffer
            *----does not exist. 
            *----Rasterize the implicit ADF line cell from bottom to top (i.e., jMin to
            *----jMax), left to right (i.e., i to iMax). For each interior sample point,
            *----determine the unsigned distance from the line segment to the sample
            *----point and combine the negative of this distance with the corresponding
            *----distance in the distance buffer.
            */
            j = jMin;
            while (j <= jMax) {
                ADF_I32      i, iMax;
                ADF_I1616 d;
                ADF_I1616 *pDist;
                

                /**
                *----Determine the column index i of the distance buffer of the first
                *----sample point that is on or to the right of the current left
                *----edge. If the current left edge intersects the current scanline at a
                *----point outside and to the left of the distance buffer, set the
                *----column index i to zero (i.e., the leftmost column of the distance
                *----buffer).
                */
                i = I1616_TO_I32(pLeft->x);
                if (pLeft->x < 0) i = 0;
                else if (pLeft->x != I32_TO_I1616(i)) i += 1;


                /**
                *----Determine the column index iMax of the distance buffer of the last
                *----sample point that is to the left of the current right edge. If the
                *----current right edge intersects the current scanline at a point
                *----outside and to the right of the distance buffer, set the column
                *----index iMax to the rightmost column of the distance buffer.
                */
                iMax = I1616_TO_I32(pRight->x);
                if (iMax >= w) iMax = w - 1;
                else if (pRight->x == I32_TO_I1616(iMax)) iMax -= 1;


                /**
                *----Determine the distance d from the line segment represented by the
                *----implicit ADF line cell to the sample point i of the current
                *----scanline
                */
                d = pLeft->d + I1616_MUL(I32_TO_I1616(i) - pLeft->x, ddx);
                

                /**
                *----Set a pointer to the entry in the distance buffer corresponding to
                *----the first sample point i of the current scanline
                */
                pDist = &dist[j * w + i];


                /**
                *----Increment the signed distance d across the current scanline to
                *----determine the distance from the line segment to each interior
                *----sample point of the current scanline. Combine each distance with
                *----its corresponding distance in the distance buffer.
                */
                while (i <= iMax) {


                    /**
                    *----Determine the negative of the magnitude of the signed distance
                    *----from the current sample point to the line segment and combine
                    *----this negative distance with its corresponding distance in the
                    *----distance buffer
                    */
                    ADF_I1616 dNeg = I1616_NEGABS(d);
                    if (dNeg > *pDist) *pDist = dNeg;


                    /**
                    *----Update the index of the current sample point, the
                    *----interpolated distance value, and the pointer to the distance
                    *----buffer corresponding to the current sample point
                    */
                    i++;
                    d += ddx;
                    pDist++;
                }


                /**
                *----Update the index of the current scanline and the left and right
                *----edge data
                */
                j++;
                if (I32_TO_I1616(j) >= pLeft->yTop) pLeft++;
                else {pLeft->x += pLeft->dxy; pLeft->d += pLeft->ddy;}
                if (I32_TO_I1616(j) >=pRight->yTop) pRight++;
                else pRight->x += pRight->dxy;
            } /* while (j <= jMax) */
        } /* else not wnb */
    } /* else diagonal line cell */
}




/**
 *-----------------------------------------------------------------------------------
 *    Process an implicit ADF corner cell for the corner whose position is stored in
 *    the specified cell data (i.e., CellData). Rasterize the implicit ADF corner cell
 *    from bottom to top, left to right. For each sample point inside the implicit ADF
 *    corner cell, compute the signed distance to the corner and combine each
 *    signed distance with the corresponding distance stored in the distance buffer.
 *    To combine the distances, the signed distance and used to replace
 *    the corresponding distance from the distance buffer if the value has a smaller 
 *    magnitude.
 *
 *    For closed-loop contours, only sample points outside all contours, i.e. those
 *    with a winding number of zero are processesd. For uniform stroke paths, both 
 *    inside and outside corner points are processed.
 *
 *    Overview
 *
 *        - Generate and render an implicit ADF corner cell using the following steps
 *            - Determine four vertices of an axis aligned square (i.e., the geometry 
 *              of the implicit ADF corner cell), where the square is centered on the 
 *              corner point and has sides of length 2*filterRad
 *            - For each cell vertex, compute the signed displacements in x and y (dx 
 *              and dy, respectively) from the cell vertex to the center of the 
 *              implicit ADF corner cell (e.g., for vertex point (x, y), dx = x - xs 
 *              and dy = y - ys)
 *            - For each scanline intersecting the implicit ADF corner cell, use a
 *              digital differential analyzer to interpolate dx and dy along the left
 *              and right edges of the implicit ADF corner cell from the displacements
 *              computed for each cell vertex, interpolate dx and dy for each sample
 *              point between the left and right edges of the cell, compute the squared
 *              distance from each sample point to the corner point as the sum of the
 *              squares of the interpolated displacements dx and dy, and compare the
 *              squared distance with the square of the corresponding distance in the
 *              distance buffer. If the squared distance from the sample point to the
 *              corner point is less than the square of the corresponding distance in
 *              the distance buffer, compute the distance from the sample
 *              point to the corner point (by taking the square root of the squared
 *              distance) and store the signed distance in
 *              the distance buffer.
 *-----------------------------------------------------------------------------------
 */

static ADF_Void ProcessCornerCell (CellData *cell, DistBuffer *distBuf, 
                                   ADF_I1616 filterRad, ADF_I8 *wnb, ADF_I8 squarecorner)
{
    ADF_I32   i, j;
    ADF_I32   w = distBuf->w;
    ADF_I32   h = distBuf->h;
    ADF_I32   iMin, jMin, iMax, jMax;
    ADF_I1616 xMin, yMin, xMax, yMax;
    ADF_I1616 sampleSpacingX = (distBuf->sampleSpacingX);
    ADF_I1616 sampleSpacingY = (distBuf->sampleSpacingY);
    ADF_I1616 invSampleSpacingX = (distBuf->invSampleSpacingX);
    ADF_I1616 invSampleSpacingY = (distBuf->invSampleSpacingY);
    ADF_I1616 *dist = distBuf->base;
    ADF_I1616 *pDist;
    ADF_I1616 dx0, dx, dy;
    ADF_I1616 xs = (cell->x0);
    ADF_I1616 ys = (cell->y0);
    ADF_I8    *pWinding;

    if (invSampleSpacingY == I1616_CONST_1)
    {
        yMax = ys + filterRad;
        yMin = ys - filterRad;
    }
    else
    {
        yMax = I1616_MUL(ys + filterRad, invSampleSpacingY);
        yMin = I1616_MUL(ys - filterRad, invSampleSpacingY);
    }

    if (invSampleSpacingX == I1616_CONST_1)
    {
        xMin = xs - filterRad;
        xMax = xs + filterRad;
    }
    else
    {
        xMin = I1616_MUL(xs - filterRad, invSampleSpacingX);
        xMax = I1616_MUL(xs + filterRad, invSampleSpacingX);
    }

    /**
     *----The implicit ADF corner cell is a square centered at the corner point with
     *----sides of length 2 * filterRad. Determine the topmost (i.e., the last)
     *----scanline (i.e., jMax) of the implicit ADF corner cell to be rasterized. The
     *----cell is not rasterized if it lies below the bottom of the distance buffer.
     *----FixedMathNote: The floating point implementation excludes the topmost
     *----horizontal edges and topmost vertices falling exactly on a scanline,
     *----whereas the fixed point implementation includes the topmost horizontal
     *----edges and topmost vertices falling exactly on a scanline. Note that samples
     *----on topmost horizontal edges and samples that fall exactly on a scanline lie
     *----at a distance filterRad from the corner and hence are mapped to a zero
     *----density. Consequently, including the topmost scanline while processing the
     *----corner cell does not affect the rendered image. Compared to the floating
     *----point implementation, the fixed point implementation omits a conditional
     *----statement and a type conversion, thereby accelerating setup calculations
     *----and improving performance at smaller PPEMs.
     */
    if (yMax < 0) return;
    jMax = (yMax >> 16);
    if (jMax >= h) jMax = h - 1;


    /**
     *----Determine the bottommost (i.e., the first) scanline (i.e., jMin) of the
     *----implicit ADF corner cell to be rasterized. FixedMathNote: The floating
     *----point implementation includes the bottommost horizontal edges and
     *----bottommost vertices falling exactly on a scanline, whereas the fixed point
     *----implementation excludes the bottommost horizontal edges and bottommost
     *----vertices falling exactly on a scanline. Note that samples on bottommost
     *----horizontal edges and samples that fall exactly on a scanline lie at a
     *----distance filterRad from the corner and hence are mapped to a zero density.
     *----Consequently, excluding the bottommost scanline while processing the corner
     *----cell does not affect the rendered image. The fixed point implementation
     *----offers two advantages over the corresponding floating point implementation:
     *----(1) a conditional statement and a type conversion are omitted, thereby
     *----accelerating setup calculations and improving performance at smaller PPEMs,
     *----and (2) the bottommost scanline is always omitted, thereby improving
     *----rasterization performance.
     */
    jMin = I1616_TO_I32(yMin);
    if (yMin < 0) jMin = 0;
    else jMin += 1;


    /**
     *----Determine the column index iMin of the distance buffer of the first sample
     *----points to the right of the left edge of the implicit ADF corner cell.
     *----FixedMathNote: The floating point implementation includes the leftmost
     *----edges and vertices falling exactly on a column, whereas the fixed point
     *----implementation excludes the leftmost edges and vertices falling exactly on
     *----a column. Note that samples on leftmost edges and samples that fall exactly
     *----on a column lie at a distance filterRad from the corner and hence are
     *----mapped to a zero density. Consequently, excluding the leftmost column while
     *----processing the corner cell does not affect the rendered image. The fixed
     *----point implementation offers two advantages over the corresponding floating
     *----point implementation: (1) a conditional statement and a type conversion are
     *----omitted, thereby accelerating setup calculations and improving performance
     *----at smaller PPEMs, and (2) the leftmost column is always omitted, thereby
     *----improving rasterization performance.
     */
    iMin = I1616_TO_I32(xMin);
    if (xMin < 0) iMin = 0;
    else iMin += 1;


    /**
     *----Determine the column index iMax of the distance buffer of the last sample
     *----points to the left of the right edge of the implicit ADF corner cell.
     *----FixedMathNote: The floating point implementation excludes the rightmost
     *----edges and vertices falling exactly on a column, whereas the fixed point
     *----implementation includes the rightmost edges and vertices falling exactly on
     *----a column. Note that samples on rightmost edges and samples that fall
     *----exactly on a column lie at a distance filterRad from the corner and hence
     *----are mapped to a zero density. Consequently, including the rightmost column
     *----while processing the corner cell does not affect the rendered image.
     *----Compared to the floating point implementation, the fixed point
     *----implementation omits a conditional statement and a type conversion, thereby
     *----accelerating setup calculations and improving performance at smaller PPEMs.
     */
    iMax = I1616_TO_I32(xMax);
    if (iMax >= w) iMax = w - 1;


    /**
     *----Determine the displacements in x and y from the corner point to the first     
     *----sample point of the bottommost scanline
     */
    if  (sampleSpacingX == I1616_CONST_1)
        dx0 = I32_TO_I1616(iMin) - xs;
    else
        dx0 = I1616_MUL(I32_TO_I1616(iMin), sampleSpacingX) - xs;
    
    if  (sampleSpacingY == I1616_CONST_1)
        dy  = I32_TO_I1616(jMin) - ys;
    else
        dy  = I1616_MUL(I32_TO_I1616(jMin), sampleSpacingY) - ys;

    if ( wnb )
    {
        int k;
        /**
        *----This is a closed loop contour path for which a winding number buffer exists.
        *----Rasterize the implicit ADF corner cell from bottom to top (i.e., jMin to
        *----jMax), left to right (i.e., iMin to iMax). Only process sample points which
        *----are outside the interior of contours as determined by the winding number buffer.
        *----For each exterior sample point, determine the distance from the corner point 
        *----to the sample point and combine this distance with the corresponding distance
        *----in the distance buffer. To improve performance, compare the squares of the
        *----distances (instead of the distances themselves) and delay the square root
        *----computation until it is needed.
        */
        pDist = &dist[iMin + w * jMin];
        pWinding = &wnb[iMin + w * jMin];

        for (j = jMin; j <= jMax; j++) 
        {
            ADF_I1616 dySqr = I1616_SQR(dy);
            ADF_I1616 absdy = ABS(dy);
            dx = dx0;
            for (k=0, i=iMin; i <= iMax; i++,k++) 
            {
                if(squarecorner && pWinding[k] == 0)
                {
                    ADF_I1616 distance = -MAX(ABS(dx),absdy);
                    if(distance > pDist[k])
                        pDist[k] = distance;
                }
                else
                {
                    ADF_I1616 distance = ABS(pDist[k]);
                    if( (distance > absdy) || (distance > ABS(dx)) )
                    {
                        ADF_I1616 dSqr = I1616_SQR(dx) + dySqr;
                        if (dSqr < I1616_SQR(distance))
                        {
                            if(pWinding[k])
                                pDist[k] = I1616_SQRT(dSqr);
                            else
                                pDist[k] = -I1616_SQRT(dSqr);
                        }
                    }
                }
                dx += sampleSpacingX;
            }
            dy += sampleSpacingY;
            pDist += w;
            pWinding += w;
        }
    } /* if ( wnb ) */

    else
    
    {
        /**
        *----This is a uniform stroke path for which a winding number buffer does not
        *----exist. All points in the corner are processed. 
        *----Rasterize the implicit ADF corner cell from bottom to top (i.e., jMin to
        *----jMax), left to right (i.e., iMin to iMax). For each interior sample point,
        *----determine the unsigned distance from the corner point to the sample point
        *----and combine the negative of this distance with the corresponding distance
        *----in the distance buffer. To improve performance, compare the squares of the
        *----distances (instead of the distances themselves) and delay the square root
        *----computation until it is needed.
        */
        for (j = jMin; j <= jMax; j++) {
            ADF_I1616 dySqr = I1616_SQR(dy);
            dx = dx0;
            pDist = &dist[iMin + w * j];
            for (i = iMin; i <= iMax; i++) {
                ADF_I1616 dSqr = I1616_SQR(dx) + dySqr;
                if (dSqr < I1616_SQR(*pDist)) *pDist = -I1616_SQRT(dSqr);
                dx += sampleSpacingX;
                pDist++;
            }
            dy += sampleSpacingY;
        }
    } /* else not wnb */
}

/**
 *-----------------------------------------------------------------------------------
 *    Rewind an internal path consisting only of flattened contours (line segments)
 */
static ADF_Void rewindPath (InternPath *path)
{
    FS_CONST ADF_U32  numPenCmds = path->numPenCmds;
    ADFPenCmdFx      *penCmds = path->penCmds;
    ADF_U32           n, ns, ne, num, i;

    /**
     *----Process all pen commands and reverse the order of the contours
     */
    ns = 0;
    for (n = 1; n < numPenCmds; n++)
    {
       if (penCmds[n].opCode == ADF_PEN_MOVETO_CMD)
       {
            /**
             *----The pen command indicates the start of a new contour
             */  
            ne = n - 1;
            num = (ne - ns)/2;
            /**
             *----Swap the first half of the contour with the second half
             */
            for (i=0; i<=num; i++)
            {
                ADF_I1616 temp;

                temp = penCmds[ns+i].x;
                penCmds[ns+i].x = penCmds[ne-i].x;
                penCmds[ne-i].x = temp;
                
                temp = penCmds[ns+i].y;
                penCmds[ns+i].y = penCmds[ne-i].y;
                penCmds[ne-i].y = temp;
            }
            ns = n;
            continue;
       }
    }

    /* last contour */
    ne = numPenCmds - 1;
    num = (ne - ns)/2;
    for (i=0; i<=num; i++)
    {
        ADF_I1616 temp;

        temp = penCmds[ns+i].x;
        penCmds[ns+i].x = penCmds[ne-i].x;
        penCmds[ne-i].x = temp;
        
        temp = penCmds[ns+i].y;
        penCmds[ns+i].y = penCmds[ne-i].y;
        penCmds[ne-i].y = temp;
    }
    return;
}

/**
 *-----------------------------------------------------------------------------------
 *    RASTERIZING A GLYPH TO SET INTERIOR DISTANCE VALUES TO POSITIVE
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    The first step of implicit ADF rendering initializes the distance buffer to be 
 *    large negative (outside) values. SetWindingNumberBuffer() computes the true winding
 *    number for each sample point in the corresponding density image. This information
 *    is used by SetGlyphInterior() to perform a binary rasterization of an
 *    implicit ADF glyph specified by the given internal path InternPath and changes
 *    the signs of distance values for sample points of the distance buffer lying
 *    inside the implicit ADF glyph from negative to positive.
 *
 *    To process each scanline in the distance buffer, SetWindingNumberBuffer() 
 *    determines x-intersections and their intersection types (i.e., ELM_POS_SLOPE or
 *    ELM_NEG_SLOPE) for each line segment of the internal path intersecting the
 *    scanline, identifies interior spans of the scanline (i.e., x-intersection pairs
 *    that demarcate interior sample points of the distance buffer). SetGlyphInterior()
 *    changes the sign of each identified interior sample point from negative to positive.
 *    The non-zero winding rule is used to identify interior spans.
 *
 *    Implementation Notes
 *
 *        - To avoid complex special-case grazing logic for handling vertices of the
 *          internal path that fall exactly on a scanline of the distance buffer (which
 *          makes the application of the non-zero winding rule more complicated),
 *          SetWindingNumberBuffer() subtracts a tiny offset to the y-coordinates of such
 *          vertices. Note that this could result in assigning an incorrect sign to
 *          sample points of the distance buffer whose distance magnitudes are smaller
 *          than the size of the offset, so the offset must be small relative to the
 *          filter radius. With the subtraction of the tiny offset, no special logic
 *          (for applying the non-zero winding rule) is required for handling
 *          horizontal segments and shared vertices of the internal path falling on a
 *          scanline because such cases will never occur. FixedMathNote: The floating
 *          point implementation subtracts a tiny offset if a vertex of the internal
 *          path falls exactly on a scanline or lies within floating point dirt of a
 *          scanline. In contrast, the fixed point implementation subtracts a tiny
 *          offset only if a vertex falls exactly on a scanline; floating point dirt
 *          does not affect the fixed point implementation.
 *        - SetWindingNumberBuffer() uses the convention that sample points located exactly
 *          on left boundaries of interior spans are included in the internal path
 *          interior, while sample points located exactly on right boundaries of
 *          interior spans are excluded from the internal path interior. Because of the
 *          subtraction of the tiny offset from the y-coordinates of vertices that fall
 *          on a scanline, bottom vertices of line segments and horizontal line
 *          segments occurring at global or local minima are included in the internal
 *          path interior, while top vertices of line segments and horizontal line
 *          segments occurring at global or local maxima are excluded from the internal
 *          path interior.
 *        - The use of a temporary winding number buffer makes sorting of
 *          x-intersections within scanlines unnecessary and simplifies the code to
 *          identify and process interior spans, thereby improving run-time
 *          performance. The winding number buffer is a buffer with the same dimensions
 *          (i.e., width and height) as the distance buffer, where each element of the
 *          winding number buffer is a signed byte initialized to 0. The winding number
 *          buffer is used in the following way. Each intersection between a path
 *          element and a scanline results in a change of +1 or -1 to the winding
 *          number (depending on the intersection type) as the scanline is traversed
 *          from left to right. It is not necessary to store the exact x-intersection
 *          with fixed point arithmetic. Instead, it suffices to note the next location
 *          (denoted L) in the distance buffer that is affected by this change in the
 *          winding number (see note below); L corresponds to the sample point in the
 *          scanline immediately following the x-intersection. Incremental changes to
 *          the winding number are computed, merged, and stored into the winding number
 *          buffer in a first pass and then used to identify interior spans in a second
 *          pass. The absolute value of the winding numbers (not transitions) are stored 
 *          in this pass.
 *        - Note that computing, merging, and storing the change in the winding number
 *          at location L has the same effect as following the convention of including
 *          sample points located exactly on left boundaries of interior spans and
 *          excluding sample points located exactly on right boundaries of interior
 *          spans (see the convention note above). Consider a sample point P located
 *          exactly on the left boundary of an interior span. Although the distance
 *          value at P is not explicitly modified from a negative value to a positive
 *          value (because the change in the winding number occurs at the location of
 *          the sample point immediately following P in the scanline), the final result
 *          is the same, because the distance value at P is zero (i.e., a value whose
 *          sign is irrelevant). Similarly, consider a sample point Q located exactly
 *          on the right boundary of an interior span. Although the distance value at Q
 *          is explicitly modified from a negative value to a positive value (because
 *          the change in the winding number occurs at the location of the sample point
 *          immediately following Q in the scanline), the final result is the same,
 *          because the distance value at Q is zero (i.e., a value whose sign is
 *          irrelevant).
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    FixedMathNote: The floating point implementation performs the binary
 *    rasterization of an implicit ADF glyph using a digital differential line
 *    rasterization algorithm. In contrast, the fixed point implementation performs the
 *    binary rasterization of an implicit ADF glyph using a fixed point version of the
 *    Bresenham line rasterization algorithm. The latter algorithm avoids an expensive
 *    division per line segment in the internal path.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
#define ELM_POS_SLOPE     1    /* Element has a positive slope at intersection point */
#define ELM_NEG_SLOPE    -1    /* Element has a negative slope at intersection point */
/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
static ADF_Bool SetWindingNumberBuffer (InternPath *path, DistBuffer *distBuf,
                                        ADF_I8 *wnb)
{
    ADF_U32          n;
    ADF_I1616        xs=0, ys=0;
    FS_CONST ADF_I32    w = distBuf->w;
    FS_CONST ADF_I32    h = distBuf->h;
    FS_CONST ADF_I1616  invSampleSpacingX = (distBuf->invSampleSpacingX);
    FS_CONST ADF_I1616  invSampleSpacingY = (distBuf->invSampleSpacingY);
    FS_CONST ADF_U32    numPenCmds = path->numPenCmds;
    ADFPenCmdFx      *penCmd = path->penCmds;
    ADF_I8           *pWinding;


    /**
     *----For each element of the internal path, determine the x-intersections
     *----between the element and the scanlines of the distance buffer intersected by
     *----the element and their corresponding x-intersection types
     */
    for (n = 0; n < numPenCmds; n++, penCmd++) {


        /**
         *----Convert the current pen position to distance buffer coordinates and 
         *----store the converted pen position in (xe, ye)
         */ 
        ADF_I1616 xe;
        ADF_I1616 ye;

        if (invSampleSpacingX == I1616_CONST_1)
            xe = penCmd->x;   
        else
           xe = I1616_MUL(penCmd->x, invSampleSpacingX);

       if ( (invSampleSpacingY == I1616_CONST_1) )
           ye = penCmd->y;
       else
           ye = I1616_MUL(penCmd->y, invSampleSpacingY);

        /**
         *----Process each pen command
         */
       if (penCmd->opCode == ADF_PEN_MOVETO_CMD)  
       {
            /**
             *----The pen command indicates the start of a new contour or the start
             *----of a new stroke skeleton. Set the previous pen position (xs, ys) to
             *----the current pen position and proceed to the next pen command.
             */
           xs = xe;
           ys = ye;
           continue;

       } else {


            /**
             *----This is a line segment. For each scanline of the distance buffer
             *----that intersects the line segment, compute the x-intersection, the
             *----x-intersection type, the change in the winding number, and the
             *----sample location at which the winding number changes. Store the
             *----change in the winding number into the winding number buffer.
             */
            ADF_I32 ix0, ix1;
            ADF_I32 j, jMin, jMax;
            ADF_I32 type;


            /**
             *----Map the minimum and maximum y-values of the line segment to
             *----integer scanlines
             */
            FS_CONST ADF_I32 iy0 = I1616_TO_I32(ys);
            FS_CONST ADF_I32 iy1 = I1616_TO_I32(ye);


            /**
             *----Determine jMin and jMax, the bottommost and topmost scanlines that
             *----intersect the line segment
             */
            if (((ys < 0) && (ye < 0)) || (iy0 == iy1)) {

                
                /**
                 *----Ignore horizontal line segments, line segments that lie below
                 *----the bottom of the distance buffer, and nearly-horizontal line
                 *----segments whose ys and ye values lie strictly between two
                 *----consecutive horizontal scanlines. Set the previous pen position
                 *----to the current pen position and proceed to the next pen
                 *----command.
                 */
                xs = xe;
                ys = ye;
                continue;


            } else if (ye > ys) {


                /**
                 *----An ascending line segment. Determine the bottommost and topmost
                 *----scanlines that intersect the line segment.
                 */
                type = ELM_POS_SLOPE;
                if (ys < 0) jMin = 0;
                else jMin = (I1616_TO_I32(ys)) + 1;
                jMax = I1616_TO_I32(ye);
                if (jMax >= h) jMax = h - 1;


            } else {


                /**
                 *----A descending line segment. Determine the bottommost and topmost
                 *----scanlines that intersect the line segment. 
                 */
                type = ELM_NEG_SLOPE;
                if (ye < 0) jMin = 0;
                else jMin = (I1616_TO_I32(ye)) + 1;
                jMax = I1616_TO_I32(ys);
                if (jMax >= h) jMax = h - 1;
            }


            /**
             *----Map the minimum and maximum x-values of the line segments to
             *----integer columns
             */
            ix0 = I1616_TO_I32(xs);
            ix1 = I1616_TO_I32(xe);
            if (ix0 == ix1) {


                /**
                 *----The line segment is vertical or nearly vertical. It affects
                 *----samples in a single column of the winding number buffer. Handle
                 *----this special case separately for better performance.
                 */
                FS_CONST ADF_I32 column = ix0 + 1;


                /**
                 *----If the line segment maps to a column that lies within the
                 *----winding number buffer, then rasterize the line segment from
                 *----bottom to top and update the winding number buffer in each
                 *----scanline
                 */
                if (column < w) {
                    pWinding = &wnb[jMin * w + column];
                    for (j = jMin; j <= jMax; j++) {
                        *pWinding += (ADF_I8)type;
                        pWinding += w;
                    }
                }


            } else {


                /**
                 *----The line segment affects samples in more than one column of the
                 *----winding number buffer. Rasterize the line segment using a fixed
                 *----point implementation of the Bresenham line rasterization
                 *----algorithm. Initialize (lx0, ly0) to (xs, ys) and initialize
                 *----(lx1, ly1) to (xe, ye).
                 */
                ADF_I1616 lx0 = xs;
                ADF_I1616 lx1 = xe;
                ADF_I1616 ly0 = ys;
                ADF_I1616 ly1 = ye;
                if (ye < ys) {


                    /**
                     *----This is a descending line segment. Swap (lx0, ly0) with
                     *----(lx1, ly1).
                     */
                    lx0 = xe;
                    lx1 = xs;
                    ly0 = ye;
                    ly1 = ys;
                }


                /**
                 *----Rasterize the ascending line segment with bottom endpoint (lx0,
                 *----ly0) and top endpoint (lx1, ly1) using a fixed point
                 *----implementation of the Bresenham line rasterization algorithm
                 */
                {
                    /**
                     *----Determine the first column of the winding number buffer
                     *----affected by the bottommost scanline of the line segment
                     */
                    FS_CONST ADF_I32 firstColumn = 1 + I1616_TO_I32(lx0);


                    /**
                     *----Determine the distance between the x coordinates of the
                     *----endpoints
                     */
                    FS_CONST ADF_I1616 dx = I1616_ABS(lx1 - lx0);


                    /**
                     *----Determine the distance between the y coordinates of the
                     *----endpoints
                     */
                    FS_CONST ADF_I1616 dy = ly1 - ly0;


                    /**
                     *----Compute the initial error. If lx1 > lx0 (i.e., the line
                     *----proceeds from left to right), then the error is the
                     *----distance between lx0 and floor(lx0). If lx1 < lx0 (i.e.,
                     *----the line segment proceeds from right to left), then the
                     *----error is the distance between lx0 and ceil(lx0). The case
                     *----lx1 == lx0 never occurs because vertical line segments are
                     *----handled separately (see above). The error is scaled by dy
                     *----to avoid divisions by dy below.
                     */
                    ADF_I1616 error = (lx1 > lx0) ? I1616_MUL(dy, (lx0 & 0xffff))
                    : (I1616_MUL(dy, I32_TO_I1616(firstColumn) - lx0));


                    /**
                     *----Compute the vertical distance (i.e., the y component of the
                     *----distance) between the bottom endpoint and the bottommost
                     *----scanline that intersects the line segment
                     */
                    FS_CONST ADF_I1616 dy0 = I32_TO_I1616(jMin) - ly0;


                    /**
                     *----Determine the step direction. If lx1 > lx0, the line
                     *----proceeds from left to right (i.e., the step direction is
                     *----positive). If lx0 < lx1, the line segment proceeds from
                     *----right to left (i.e., the step direction is negative). The
                     *----case lx1 == lx0 never occurs because vertical line segments
                     *----are handled separately (see above).
                     */
                    FS_CONST ADF_I32 iStep = (lx1 > lx0) ? 1 : -1;


                    /**
                     *----Initialize the column index i to the integer column in the
                     *----winding number buffer corresponding to the first endpoint
                     *----of the line segment
                     */
                    ADF_I32 i = firstColumn;


                    /**
                     *----Set a pointer to the winding number buffer
                     */
                    pWinding = &wnb[jMin * w];


                    /**
                     *----Update the initial error. The true error that results from
                     *----advancing from scanline y to scanline y+1 is dx/dy. The
                     *----scaled error (i.e., after multiplying by dy) is dx.
                     *----Therefore, the scaled error that results from moving from y
                     *----= ly0 up to the bottommost scanline jMin is dx * dy0.
                     */
                    error += I1616_MUL(dx, dy0);


                    /**
                     *----Rasterize the line segment from bottom to top
                     */
                    for (j = jMin; j <= jMax; ++j) {

                        
                        /**
                         *----While the scaled error exceeds dy (i.e., the error
                         *----exceeds the mathematical value dx/dy), update the
                         *----column index i and subtract dy from the error
                         */
                        while (error > dy) {
                            error -= dy;
                            i += iStep;
                        }


                        /**
                         *----If the column index lies within the winding number
                         *----buffer, update the winding number buffer
                         */
                        if (i < 0) i = 0;
                        if (i < w) pWinding[i] += (ADF_I8)type;


                        /**
                         *----Update the scaled error (i.e., the scaled error
                         *----introduced by advancing from scanline y to scanline y+1
                         *----is dx)
                         */
                        error += dx;


                        /**
                         *----Advance the pointer to the winding number buffer to the
                         *----next scanline
                         */
                        pWinding += w;
                    }
                }
            }
        }


        /**
         *----Set the previous pen position to the current pen position and proceed
         *----to the next pen command
         */
        xs = xe;
        ys = ye;
    }

    /**
     *----Traverse the winding number buffer from bottom to top, left to right, and 
     *----convert transitions to the absolute value of actual winding numbers
     */
    {
        ADF_I32          row, col;
        ADF_I1616        *pDist; 
        ADF_Bool         overlapping_contour = 0;
        ADF_I32          need_rewind = 0;
        pWinding = wnb;
        row = h;
        pDist = distBuf->base; 
        while(row--)
        {

            /**
            *----Initialize the winding number to 0 at the beginning of each scanline
            */
            ADF_I8 windingNumber = 0;
            col = w;
            while(col--)
            {

                /**
                *----Update the winding number before processing the current sample in
                *----the scanline
                */
                windingNumber += *pWinding;
                /* check if rewind needed. Only rewind if entire glyph is mis-wound */
                if(windingNumber > 0) 
                    need_rewind = 1; /* do not rewind */
                else if(windingNumber < 0 && need_rewind == 0)
                    need_rewind = 2; /* possibly rewind */

                *pWinding = ADF_ABS(windingNumber);
                if (*pWinding)
                {
                    *pDist = LARGE_INSIDE_DIST_VAL;
                    if ((*pWinding != 1) && (overlapping_contour == 0))
                        overlapping_contour = 1;
                }
                else
                    *pDist = LARGE_OUTSIDE_DIST_VAL;

                pWinding++;
                pDist++;

            }
        }

        /* if entire contour is mis-wound, rewind the internal path */
        if(need_rewind==2)
            rewindPath(path);

        return overlapping_contour;
    }
}

/**
 *-----------------------------------------------------------------------------------
 *    MAPPING A DISTANCE BUFFER TO A DENSITY IMAGE
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Map distances in a specified distance buffer to density values, perform color
 *    reduction on these density values if enabled, and pack the final density values
 *    into pixels of a given density image. The packing depends on the specified
 *    display mode. For example, when the display mode is ADF_REND_MODE_CRT (i.e., for
 *    CRT rendering), one density value is packed into each density image pixel. When
 *    the display mode is ADF_REND_MODE_RGBv or ADF_REND_MODE_BGRv (two forms of LCD
 *    rendering), three adjacent density values and an alpha value derived from these
 *    three density values are packed into each density image pixel.
 *
 *    Note that the convention in this implementation places a sample point
 *    corresponding to the center of a density image pixel with integer pixel
 *    coordinates (i, j) at the ADF_I1616 fixed point image position (x, y) =
 *    (I32_TO_I1616(i), I32_TO_I1616(j)). In the distance buffer, a sample point
 *    indexed by the distance buffer coordinates (ii, jj) corresponds to the position
 *    (ii * invSampleSpacingX, jj * invSampleSpacingY) in ADF_I1616 fixed point image
 *    coordinates. Thus, the center of the density image pixel (i, j) maps to the
 *    distance buffer sample (i, j) for a display mode indicating CRT rendering, to the
 *    distance buffer sample (3i, j) for a display mode indicating LCD rendering with
 *    vertical stripes, and to the distance buffer sample (i, 3j) for a display mode
 *    indicating LCD rendering with horizontal stripes.
 *
 *    The color reduction algorithm computes a weighted average of the RGB color
 *    components, finds the absolute difference between each color component and the
 *    weighted average, and sums these component differences to obtain a single color
 *    difference value. If the color difference value is above a specified threshold,
 *    the algorithm desaturates the pixel by moving each color component towards the
 *    weighted average. The amount of adjustment to the color components depends on how
 *    far the color difference value lies above the threshold.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    DevNote: Display modes ADF_REND_MODE_BGRv, ADF_REND_MODE_RGBh, and 
 *    ADF_REND_MODE_BGRh are currently unsupported.
 *-----------------------------------------------------------------------------------
 *    FixedMathNote: Both the floating point and fixed point implementations use the
 *    precomputed variables fScale and fOffset to map distance values in the range
 *    [minDist,maxDist) to density values in the range [0,1). fScale is computed as (1
 *    / (maxDist - minDist)) and fOffset is computed as (-minDist * fScale). The linear
 *    map from a distance value to a density value is computed as density = (fScale *
 *    distance + fOffset). The purpose of precomputing fScale and fOffset is to avoid
 *    expensive per-sample arithmetic. Whereas the floating point implementation is
 *    relatively straightforward and easy to understand, the fixed point implementation
 *    contains several subtleties discussed below.
 *
 *    In the fixed point implementation, maxDist and minDist are represented as
 *    ADF_I1616 fixed point values. If (maxDist - minDist) is less than 3 (i.e.,
 *    (maxDist - minDist) is less than or equal to the mathematical value (2.0 /
 *    65536.0)), the reciprocal will have a mathematical value that is at least
 *    (65536.0 / 2.0) = 32768.0, which overflows the ADF_I1616 fixed point
 *    representation. In this case, maxDist and minDist are both set to the arithmetic
 *    mean of maxDist and minDist (i.e., avgDist = (maxDist + minDist) / 2). This step
 *    effectively forces a binary rasterization of the glyph: sample points whose
 *    distance values are less than or equal to avgDist will be assigned density values
 *    of 0 (i.e., no density), and all other sample points will be assigned density
 *    values of 255 (i.e., maximum density). Note that the fixed point implementation
 *    below uses neither fScale nor fOffset when maxDist is equal to minDist.
 *
 *    Since the computed density value lies in the range [0,1), the fixed point
 *    implementation optimizes the linear map (fScale * distance + fOffset) by
 *    discarding the integer bits as follows:
 *
 *      - fScale is computed and stored as an ADF_I1616 fixed point value (i.e., the
 *        data type of fScale is ADF_I1616).
 *
 *      - fOffset is computed as the product of two ADF_I1616 fixed point values (i.e.,
 *        -minDist * fScale) and is therefore a 64-bit 32.32 fixed point value.
 *        However, only the fractional bits of this product (i.e., the low 32 bits) are
 *        stored in fOffset (i.e., the data type of fOffset is ADF_U0032).
 *
 *      - When computing the linear map (fScale * distance + fOffset), the product
 *        (fScale * distance) is a product of two ADF_I1616 fixed point values, which
 *        is a 64-bit 32.32 fixed point value. However, only the fractional bits of
 *        this product (i.e., the low 32 bits) are kept and added to the ADF_U0032
 *        fixed point value fOffset. The resulting sum is an ADF_U0032 fixed point
 *        density value that lies in the range [0,1). The discarded integer bits do not
 *        affect the computation of the fractional bits.
 *
 *    This approach has two main advantanges over using ADF_I1616 fixed point
 *    arithmetic throughout the entire computation: 1) fewer operations are required,
 *    which implies faster performance, and 2) fractional precision is maximized in the
 *    computed density value.
 *
 *    Since this fixed implementation ignores the integer bits when computing the
 *    linear map (fScale * distance + fOffset), it is critical that the computed result
 *    truly lies in the range [0,1). If the computed result could become negative, for
 *    example, then the fractional bits in the ADF_U0032 fixed point result would "wrap
 *    around" (i.e., instead of being all zeroes, they would become all ones), thereby
 *    causing samples that ought to have low density values to have high density
 *    values. Similarly, if the computed result could become 1 or greater, then the
 *    fractional bits in the ADF_U0032 fixed point result would "wrap around" (i.e.,
 *    instead of being all ones, they would become all zeroes), thereby causing samples
 *    that ought to have high density values to have low density values.
 *
 *    The following is a proof that neither case can occur. Let dMin be minDist (i.e.,
 *    the smallest distance value in the range [minDist,maxDist)). It follows that
 *
 *        dMin * fScale + fOffset = minDist * fScale + fOffset                      
 *                                = minDist * fScale + (-minDist * fScale)          
 *                                = (minDist - minDist) * fScale                      
 *                                = 0
 *
 *    Let dMax be (maxDist - 1) (i.e., the largest distance value in the range
 *    [minDist,maxDist)). It follows that
 *
 *        dMax * fScale + fOffset =  (maxDist - 1) * fScale + fOffset
 *                                =  (maxDist - 1) * fScale + (-minDist * fScale)
 *                                =  (maxDist - minDist - 1) * fScale
 *                                <= (maxDist - minDist - 1) / (maxDist - minDist) ($)
 *                                <  (maxDist - minDist) / (maxDist - minDist)
 *                                =  1
 *
 *    The inequality in the line marked with ($) is due to the fact that fScale is
 *    computed using the I1616_DIV() function, which rounds the computed quotient
 *    towards zero (see ADFFixedMath.h for more information about the I1616_DIV()
 *    function). Therefore, fScale is less than or equal to the true mathematical value
 *    of (1 / (maxDist - minDist)).
 *
 *    The derivations above show that the smallest distance value (i.e., minDist) maps
 *    to zero and that the largest distance value (i.e., maxDist - 1) maps to a value
 *    less than 1. Since (d * fScale + fOffset) is a linear transformation, it follows
 *    that each value in the range [minDist, maxDist) is mapped to a value in the
 *    range [0, 1).
 *-----------------------------------------------------------------------------------
 *    INTEGER RGB-TO-YUV TABLES (Y-COMPONENT ONLY)
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    The color reduction algorithm converts integer RGB values in the range [0,255] to
 *    YUV. The Y component can be expressed as a linear combination of the RGB values
 *    (i.e., Y = (kr * R) + (kg * G) + (kb * B), where kr, kg, and kb are floating
 *    point constants whose sum is 1). The implementation below uses lookup tables to
 *    avoid conversions between integer and fixed point values. These lookup tables
 *    are:
 *
 *    yuvRedTable: Maps R to (kr * R), where kr is 0.299. The index R must be in the
 *    range [0,255].
 *
 *    yuvGreenTable: Maps G to (kg * G), where kg is 0.587. The index G must be in the
 *    range [0,255].
 *
 *    yuvBlueTable: Maps B to (kb * B), where kb is 0.114. The index B must be in the
 *    range [0,255].
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
static FS_CONST ADF_I32 yuvRedTable[] = {
    0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01, 0x02, 
    0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x04, 0x04, 
    0x04, 0x05, 0x05, 0x05, 0x05, 0x06, 0x06, 0x06, 
    0x07, 0x07, 0x07, 0x08, 0x08, 0x08, 0x08, 0x09, 
    0x09, 0x09, 0x0a, 0x0a, 0x0a, 0x0b, 0x0b, 0x0b, 
    0x0b, 0x0c, 0x0c, 0x0c, 0x0d, 0x0d, 0x0d, 0x0e, 
    0x0e, 0x0e, 0x0e, 0x0f, 0x0f, 0x0f, 0x10, 0x10, 
    0x10, 0x11, 0x11, 0x11, 0x11, 0x12, 0x12, 0x12, 
    0x13, 0x13, 0x13, 0x14, 0x14, 0x14, 0x14, 0x15, 
    0x15, 0x15, 0x16, 0x16, 0x16, 0x17, 0x17, 0x17, 
    0x17, 0x18, 0x18, 0x18, 0x19, 0x19, 0x19, 0x1a, 
    0x1a, 0x1a, 0x1a, 0x1b, 0x1b, 0x1b, 0x1c, 0x1c, 
    0x1c, 0x1d, 0x1d, 0x1d, 0x1d, 0x1e, 0x1e, 0x1e, 
    0x1f, 0x1f, 0x1f, 0x1f, 0x20, 0x20, 0x20, 0x21, 
    0x21, 0x21, 0x22, 0x22, 0x22, 0x22, 0x23, 0x23, 
    0x23, 0x24, 0x24, 0x24, 0x25, 0x25, 0x25, 0x25, 
    0x26, 0x26, 0x26, 0x27, 0x27, 0x27, 0x28, 0x28, 
    0x28, 0x28, 0x29, 0x29, 0x29, 0x2a, 0x2a, 0x2a, 
    0x2b, 0x2b, 0x2b, 0x2b, 0x2c, 0x2c, 0x2c, 0x2d, 
    0x2d, 0x2d, 0x2e, 0x2e, 0x2e, 0x2e, 0x2f, 0x2f, 
    0x2f, 0x30, 0x30, 0x30, 0x31, 0x31, 0x31, 0x31, 
    0x32, 0x32, 0x32, 0x33, 0x33, 0x33, 0x34, 0x34, 
    0x34, 0x34, 0x35, 0x35, 0x35, 0x36, 0x36, 0x36, 
    0x37, 0x37, 0x37, 0x37, 0x38, 0x38, 0x38, 0x39, 
    0x39, 0x39, 0x3a, 0x3a, 0x3a, 0x3a, 0x3b, 0x3b, 
    0x3b, 0x3c, 0x3c, 0x3c, 0x3c, 0x3d, 0x3d, 0x3d, 
    0x3e, 0x3e, 0x3e, 0x3f, 0x3f, 0x3f, 0x3f, 0x40, 
    0x40, 0x40, 0x41, 0x41, 0x41, 0x42, 0x42, 0x42, 
    0x42, 0x43, 0x43, 0x43, 0x44, 0x44, 0x44, 0x45, 
    0x45, 0x45, 0x45, 0x46, 0x46, 0x46, 0x47, 0x47, 
    0x47, 0x48, 0x48, 0x48, 0x48, 0x49, 0x49, 0x49, 
    0x4a, 0x4a, 0x4a, 0x4b, 0x4b, 0x4b, 0x4b, 0x4c, 
};
/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
static FS_CONST ADF_I32 yuvGreenTable[] = {
    0x00, 0x00, 0x01, 0x01, 0x02, 0x02, 0x03, 0x04, 
    0x04, 0x05, 0x05, 0x06, 0x07, 0x07, 0x08, 0x08, 
    0x09, 0x09, 0x0a, 0x0b, 0x0b, 0x0c, 0x0c, 0x0d, 
    0x0e, 0x0e, 0x0f, 0x0f, 0x10, 0x11, 0x11, 0x12, 
    0x12, 0x13, 0x13, 0x14, 0x15, 0x15, 0x16, 0x16, 
    0x17, 0x18, 0x18, 0x19, 0x19, 0x1a, 0x1b, 0x1b, 
    0x1c, 0x1c, 0x1d, 0x1d, 0x1e, 0x1f, 0x1f, 0x20, 
    0x20, 0x21, 0x22, 0x22, 0x23, 0x23, 0x24, 0x24, 
    0x25, 0x26, 0x26, 0x27, 0x27, 0x28, 0x29, 0x29, 
    0x2a, 0x2a, 0x2b, 0x2c, 0x2c, 0x2d, 0x2d, 0x2e, 
    0x2e, 0x2f, 0x30, 0x30, 0x31, 0x31, 0x32, 0x33, 
    0x33, 0x34, 0x34, 0x35, 0x36, 0x36, 0x37, 0x37, 
    0x38, 0x38, 0x39, 0x3a, 0x3a, 0x3b, 0x3b, 0x3c, 
    0x3d, 0x3d, 0x3e, 0x3e, 0x3f, 0x3f, 0x40, 0x41, 
    0x41, 0x42, 0x42, 0x43, 0x44, 0x44, 0x45, 0x45, 
    0x46, 0x47, 0x47, 0x48, 0x48, 0x49, 0x49, 0x4a, 
    0x4b, 0x4b, 0x4c, 0x4c, 0x4d, 0x4e, 0x4e, 0x4f, 
    0x4f, 0x50, 0x51, 0x51, 0x52, 0x52, 0x53, 0x53, 
    0x54, 0x55, 0x55, 0x56, 0x56, 0x57, 0x58, 0x58, 
    0x59, 0x59, 0x5a, 0x5a, 0x5b, 0x5c, 0x5c, 0x5d, 
    0x5d, 0x5e, 0x5f, 0x5f, 0x60, 0x60, 0x61, 0x62, 
    0x62, 0x63, 0x63, 0x64, 0x64, 0x65, 0x66, 0x66, 
    0x67, 0x67, 0x68, 0x69, 0x69, 0x6a, 0x6a, 0x6b, 
    0x6c, 0x6c, 0x6d, 0x6d, 0x6e, 0x6e, 0x6f, 0x70, 
    0x70, 0x71, 0x71, 0x72, 0x73, 0x73, 0x74, 0x74, 
    0x75, 0x75, 0x76, 0x77, 0x77, 0x78, 0x78, 0x79, 
    0x7a, 0x7a, 0x7b, 0x7b, 0x7c, 0x7d, 0x7d, 0x7e, 
    0x7e, 0x7f, 0x7f, 0x80, 0x81, 0x81, 0x82, 0x82, 
    0x83, 0x84, 0x84, 0x85, 0x85, 0x86, 0x87, 0x87, 
    0x88, 0x88, 0x89, 0x89, 0x8a, 0x8b, 0x8b, 0x8c, 
    0x8c, 0x8d, 0x8e, 0x8e, 0x8f, 0x8f, 0x90, 0x90, 
    0x91, 0x92, 0x92, 0x93, 0x93, 0x94, 0x95, 0x95, 
};
/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
static FS_CONST ADF_I32 yuvBlueTable[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 
    0x01, 0x01, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 
    0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x03, 
    0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 
    0x04, 0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0x05, 
    0x05, 0x05, 0x05, 0x05, 0x05, 0x06, 0x06, 0x06, 
    0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x07, 0x07, 
    0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x08, 
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x09, 
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 
    0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 0x0a, 
    0x0a, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 
    0x0b, 0x0b, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 
    0x0c, 0x0c, 0x0c, 0x0d, 0x0d, 0x0d, 0x0d, 0x0d, 
    0x0d, 0x0d, 0x0d, 0x0e, 0x0e, 0x0e, 0x0e, 0x0e, 
    0x0e, 0x0e, 0x0e, 0x0e, 0x0f, 0x0f, 0x0f, 0x0f, 
    0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x10, 0x10, 0x10, 
    0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x11, 0x11, 
    0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x12, 0x12, 
    0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x12, 0x13, 
    0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 0x13, 
    0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 0x14, 
    0x14, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 0x15, 
    0x15, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 0x16, 
    0x16, 0x16, 0x17, 0x17, 0x17, 0x17, 0x17, 0x17, 
    0x17, 0x17, 0x17, 0x18, 0x18, 0x18, 0x18, 0x18, 
    0x18, 0x18, 0x18, 0x18, 0x19, 0x19, 0x19, 0x19, 
    0x19, 0x19, 0x19, 0x19, 0x19, 0x1a, 0x1a, 0x1a, 
    0x1a, 0x1a, 0x1a, 0x1a, 0x1a, 0x1b, 0x1b, 0x1b, 
    0x1b, 0x1b, 0x1b, 0x1b, 0x1b, 0x1b, 0x1c, 0x1c, 
    0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1c, 0x1d, 
};
/**
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 */
static ADF_Void MapDistBufferToImage (DistBuffer *dist, ADFRenderGlyphData
*renderGlyphData, ADF_U32 displayMode, ADFImage *image)
{
    ADF_U32 i, j;
    ADF_U32 w = (ADF_U32) image->w;
    ADF_U32 h = (ADF_U32) image->h;


    /**
     *----Extract the inside cutoff, outside cutoff, and gamma CSM parameters. The
     *----outside cutoff is the "minimum distance" (i.e., a sample point whose
     *----distance value is less than or equal to the minimum distance will be
     *----assigned a density value of 0). The inside cutoff is the "maximum distance"
     *----(i.e., a sample point whose distance value is greater than or equal to the
     *----maximum distance will be assigned a density value of 255). Convert all
     *----parameters from floating point image coordinates to ADF_I1616 fixed point
     *----image coordinates.
     */
    ADF_I1616 maxDist = FLOAT_TO_I1616(renderGlyphData->insideCutoff);
    ADF_I1616 minDist = FLOAT_TO_I1616(renderGlyphData->outsideCutoff);
    ADF_I1616 gamma = FLOAT_TO_I1616(renderGlyphData->gamma);


    /**
     *----The uninitialized parameters needed to map a distance value in the range
     *----[minDist,maxDist) to a density value in the range [0,1)
     */
    ADF_I1616 fScale=0;
    ADF_U0032 fOffset=0;


    /**
     *----FixedMathNote: Determine if the reciprocal of (maxDist - minDist) will
     *----overflow the ADF_I1616 fixed point representation
     */
    if ((maxDist - minDist) < 3) {


        /**
         *----FixedMathNote: The reciprocal of (maxDist - minDist) overflows the
         *----ADF_I1616 fixed point representation (see the fixed point math note in
         *----the header of this function for a detailed explanation). Set both
         *----maxDist and minDist to avgDist (i.e., the arithmetic mean of maxDist
         *----and minDist). Note that fScale and fOffset are not used when maxDist is
         *----equal to minDist.
         */
        ADF_I1616 avgDist = (maxDist + minDist) >> 1;
        maxDist = minDist = avgDist;

        
    } else {


        /**
         *----FixedMathNote: The reciprocal of (maxDist - minDist) does not overflow
         *----the ADF_I1616 fixed point representation (see the fixed point math note
         *----in the header of this function for a detailed explanation). Compute the
         *----parameters needed to map a distance value in the range
         *----[minDist,maxDist) to a density value in the range [0,1). The linear map
         *----is performed as follows: density = fScale * distance + fOffset. The
         *----parameters fScale and fOffset are precomputed here to avoid expensive
         *----per-sample arithmetic in the inner loops below.
         */
        ADF_I32 divStatus;
        fScale = I1616_DIV(I1616_CONST_1, maxDist - minDist, &divStatus);
        fOffset = (ADF_U0032) (-minDist * fScale);
    }


    /**
     *----Map distances and pack pixels based on the specified display mode. Color
     *----reduction is performed on LCD display modes if enabled.
     */
    switch (displayMode) {


        /**
         *----CRT rendering: map distances to density values and pack one density 
         *----value per density image pixel
         */
        case ADF_REND_MODE_CRT:
        {
            ADF_U32   imageSize = w * h;
            ADF_U8    *pPixel = image->base;
            ADF_I1616 *pDist = dist->base;
            i = imageSize;
            while(i--)
            {

                /**
                 *----Initialize the density value for the current sample point to
                 *----255 (i.e., maximum density), which represents the case where
                 *----the sample point is inside the shape and outside the filter
                 *----radius
                 */
                ADF_U32 density = 255;


                /**
                 *----Read the current sample point's distance value from the
                 *----distance buffer. If the distance value is less than or equal to
                 *----the minimum distance, then the current sample point is outside
                 *----the shape and outside the filter radius, so set its density
                 *----value to 0 (i.e., no density). Otherwise, if the distance is
                 *----within the filter radius, map the distance to a density value
                 *----using the precomputed fScale and fOffset parameters and then
                 *----perform the requested gamma correction.
                 */
                ADF_I1616 Dist = *pDist++;
                if (Dist <= minDist) density = 0;
                else if (Dist < maxDist) {


                    /**
                     *----Normalize the distance value to an ADF_U0032 fixed point
                     *----value that lies in the mathematical range (0, 1).
                     */
                    ADF_U0032 f = ((ADF_U0032) (Dist * fScale)) + fOffset;


                    /**
                     *----Determine if gamma is equal to 1
                     */
                    if (gamma == I1616_CONST_1) {

                        
                        /**
                         *----Gamma is 1, so the gamma correction step can be
                         *----skipped. Map the normalized ADF_U0032 fixed point
                         *----distance value to an integer density value in the range
                         *----[0, 255].
                         */
                        density = U0032_TO_U8(f);

                        
                    } else {


                        /**
                         *----Gamma is not equal to 1. Convert f from an ADF_U0032
                         *----fixed point value to an ADF_I1616 fixed point value by
                         *----shifting f to the right by 16 bits, perform the
                         *----requested gamma correction, and map the normalized and
                         *----gamma-corrected ADF_I1616 fixed point distance value to
                         *----an integer density value in the range [0, 255].
                         */
                        ADF_I1616 d = I1616_POW01(f >> 16, gamma);
                        density = I1616_TO_U8(d);
                    }
                }


                /**
                 *----Write the density value to the density image pixel
                 */
                *pPixel++ = (ADF_I8)density;
            }
            break;
        }


        /**
         *----Vertical striping - subpixels are vertical and arranged horizontally
         *----RGBv rendering: map distances to density values, apply color reduction
         *----if enabled, and pack the final density values and a derived alpha value
         *----into each density image pixel
         *----BGRv rendering: same as RGBv except R and B are swapped
         *----interprete R->B, B->R everywhere in this case 
         *----color reduction is not currently supported for BGRv
         */
        case ADF_REND_MODE_RGBv:
        case ADF_REND_MODE_BGRv:
        {
            ADF_U32   *pPixel = (ADF_U32 *) image->base;
            ADF_I1616 *pDist = dist->base;
            ADF_U32   maxDensity;
                
                
            /**
             *----Extract the Boolean that determines whether or not to apply color
             *----reduction. The Boolean value is constant across all pixels of a
             *----glyph.
             */
            ADF_U32 useColorReduction = renderGlyphData->useColorReduction;


            /**
             *----Convert the user-specified color reduction amount to a color
             *----threshold as follows. The color reduction amount is a floating
             *----point value that lies in the range [0,1]. Subtract this value from
             *----1 and multiply the difference by a factor of 0.25 (i.e., (1.0f -
             *----renderGlyphData->colorReductionAmt) * 0.25f). The result is a
             *----threshold in the range [0,0.25], where a value of zero corresponds
             *----to maximum color reduction (i.e., the pixel has no color at all)
             *----and a value of 0.25 corresponds to minimum color reduction. The
             *----factor of 0.25 has been determined experimentally to be a good
             *----compromise between good antialiasing quality and a lack of
             *----perceived color fringes. That is, a smaller factor implies less
             *----color at the expense of antialiasing, and vice versa. The color
             *----reduction implementation (see below) processes integer density
             *----values in the range [0,255] instead of floating point density
             *----values in the range [0,1], so the factor of 0.25 is scaled by 256
             *----to obtain a final factor of 64. The color threshold is constant
             *----across all pixels of a glyph and must be an integer in the range
             *----[0,64]. Color reduction desaturates an image pixel if this
             *----threshold is exceeded. FixedMathNode: Convert the color reduction
             *----amount from a floating point value to an ADF_I1616 fixed point
             *----value and subtract the result from 1 (i.e., I1616_CONST_1).
             *----Multiply the result by 64 and convert the product from an ADF_I1616
             *----fixed point value to a signed integer. Multiplication by 64 is
             *----equivalent to a left shift of 6 bits, and conversion from a
             *----non-negative ADF_I1616 fixed point value to a signed integer is
             *----equivalent to a right shift of 16 bits. Therefore, these two steps
             *----can be combined into a single right shift of 10 bits.
             */
            FS_CONST ADF_I32 colorThreshold = (I1616_CONST_1 - FLOAT_TO_I1616(
            renderGlyphData->colorReductionAmt)) >> 10;
      

            /**
             *----Disable color reduction for ADF_REND_MODE_BGRv since unsupported
             */
            if(displayMode == ADF_REND_MODE_BGRv) useColorReduction = 0;


            /**
             *----Process the distance buffer from bottom to top, left to right
             */
            for (j = 0; j < h; j++) {


                /**
                 *----For LCD rendering with vertical RGB stripes, the green
                 *----component of density image pixel (i, j) is determined from the
                 *----distance buffer sample indexed by (3i, j), and the red and blue
                 *----components are indexed by (3i - 1, j) and (3i + 1, j),
                 *----respectively. Therefore, the red component of each leftmost
                 *----pixel in the density image, which falls outside the density
                 *----buffer, is set to zero density and the rightmost samples of the
                 *----distance buffer (which also correspond to red components) are
                 *----computed but not used. See description of distance buffers above.
                 */
                ADF_U32 G, B;
                ADF_U32 R = 0;
                for (i = 0; i < w; i++) {


                    /**
                     *----Extract the green, blue, and red distance values from the
                     *----distance buffer. Note that the red distance value (i.e.,
                     *----distR) corresponds to the red sample point for the next
                     *----density image pixel (i.e., the pixel with index (i + 1,
                     *----j)), not the current density image pixel. The red distance
                     *----value is mapped to a density value (i.e., R) at the bottom
                     *----of this loop (see below) and is propagated to the next
                     *----loop iteration.
                     */
                    ADF_I1616 distG = *pDist++;
                    ADF_I1616 distB = *pDist++;
                    ADF_I1616 distR = *pDist++;


                    /**
                     *----Three cases are possible: (1) the red, green, and blue
                     *----distance values are all less than the minimum distance, (2)
                     *----the red, green, and blue distance values are all greater
                     *----than the maximum distance, or (3) at least one of the red,
                     *----green, or blue distance values is in the range
                     *----[minDist,maxDist). Since the first two cases are common and
                     *----are unaffected by color reduction, handle them separately
                     *----to improve performance. As noted above, the red distance
                     *----value (i.e., distR) corresponds to the next density image
                     *----pixel, not the current density image pixel. The red
                     *----distance value for the current density image pixel is less
                     *----than the minimum distance (i.e., minDist) if and only if
                     *----the red density value (i.e., R) is zero. Similarly, the red
                     *----distance value for the current density image pixel is
                     *----greater than the maximum distance (i.e., maxDist) if and
                     *----only if the red density value (i.e., R) is 255.
                     */
                    if (R == 0 && (distG <= minDist) && (distB <= minDist)) {


                        /**
                         *----The distances associated with all three sample points
                         *----are all less than the minimum distance, so the density
                         *----value assigned to each sample point is zero. Skip color
                         *----reduction because it has no effect on a pixel whose RGB
                         *----values are equal. Each component of each pixel of the
                         *----density image is cleared to zero in RenderGlyph(),
                         *----so it suffices to advance the density image pointer to
                         *----the next pixel.
                         */
                        pPixel++;

                            
                    } else if (R == 255 && (distG >= maxDist) &&
                    (distB >= maxDist)) {


                        /**
                         *----The distances associated with all three sample points
                         *----are all greater than the maximum distance, so the
                         *----density value assigned to each sample point is 255.
                         *----Skip color reduction because it has no effect on a
                         *----pixel whose RGB values are equal. The maximum density
                         *----value is also 255, so set the pixel value to RGBA =
                         *----(255,255,255,255) and advance the pointer to the next
                         *----pixel.
                         */
                        *pPixel++ = 0xFFFFFFFF;

                            
                    } else {


                        /**
                         *----At least one of the distances associated with the red,
                         *----green, or blue sample points is in the range
                         *----[minDist,maxDist). Initialize the density values of the
                         *----green and blue components to 255 (i.e., maximum
                         *----density), which represents the case where the green and
                         *----blue sample points are inside the shape and outside the
                         *----filter radius.
                         */
                        G = 255;
                        B = 255;


                        /**
                         *----Map the green distance value to a density value. If the
                         *----distance value is less than or equal to the minimum
                         *----distance, then the corresponding sample point is
                         *----outside the shape and outside the filter radius, so set
                         *----its density value to 0 (i.e., no density). Otherwise,
                         *----if the distance is within the filter radius, apply the
                         *----requested gamma correction and map the distance to a
                         *----density value using the precomputed fScale and fOffset
                         *----parameters.
                         */
                        if (distG <= minDist) G = 0;
                        else if (distG < maxDist) {


                            /**
                             *----Normalize the green distance value to an ADF_U0032
                             *----fixed point value that lies in the mathematical
                             *----range (0, 1).
                             */
                            ADF_U0032 f = ((ADF_U0032) (distG * fScale)) + fOffset;


                            /**
                             *----Determine if gamma is equal to 1
                             */
                            if (gamma == I1616_CONST_1) {

                        
                                /**
                                 *----Gamma is 1, so the gamma correction step can be
                                 *----skipped. Map the normalized ADF_U0032 fixed
                                 *----point distance value to an integer density
                                 *----value in the range [0, 255].
                                 */
                                G = U0032_TO_U8(f);


                            } else {


                                /**
                                 *----Gamma is not equal to 1. Convert f from an
                                 *----ADF_U0032 fixed point value to an ADF_I1616
                                 *----fixed point value by shifting f to the right by
                                 *----16 bits, perform the requested gamma
                                 *----correction, and map the normalized and
                                 *----gamma-corrected ADF_I1616 fixed point distance
                                 *----value to an integer density value in the range
                                 *----[0, 255].
                                 */
                                ADF_I1616 d = I1616_POW01(f >> 16, gamma);
                                G = I1616_TO_U8(d);
                            }
                        }


                        /**
                         *----Map the blue distance value to a density value using
                         *----the same method described above for the green component
                         */
                        if (distB <= minDist) B = 0;
                        else if (distB < maxDist) {


                            /**
                             *----Normalize the blue distance value to an ADF_U0032
                             *----fixed point value that lies in the mathematical
                             *----range (0, 1).
                             */
                            ADF_U0032 f = ((ADF_U0032) (distB * fScale)) + fOffset;
                            

                            /**
                             *----Determine if gamma is equal to 1
                             */
                            if (gamma == I1616_CONST_1) {

                                
                                /**
                                 *----Gamma is 1, so the gamma correction step can be
                                 *----skipped. Map the normalized ADF_U0032 fixed
                                 *----point distance value to an integer density
                                 *----value in the range [0, 255].
                                 */
                                B = U0032_TO_U8(f);
                                

                            } else {


                                /**
                                 *----Gamma is not equal to 1. Convert f from an
                                 *----ADF_U0032 fixed point value to an ADF_I1616
                                 *----fixed point value by shifting f to the right by
                                 *----16 bits, perform the requested gamma
                                 *----correction, and map the normalized and
                                 *----gamma-corrected ADF_I1616 fixed point distance
                                 *----value to an integer density value in the range
                                 *----[0, 255].
                                 */
                                ADF_I1616 d = I1616_POW01(f >> 16, gamma);
                                B = I1616_TO_U8(d);
                            }
                        }


                        /**
                         *----Determine if the red, green, and blue density values
                         *----are equal
                         */
                        if (R != G || G != B) {


                            /**
                             *----The three density values are not equal. Perform
                             *----color reduction if enabled and determine the
                             *----maximum density value.
                             */
                            if (useColorReduction) {


                                /**
                                 *----Map RGB to YUV using table lookups, but compute
                                 *----only the Y component. This is a weighted
                                 *----averaging technique that weights green the
                                 *----highest and blue the least.
                                 */
                                ADF_I32 Y = yuvRedTable[R] + yuvGreenTable[G] +
                                yuvBlueTable[B];


                                /**
                                 *----Compute the absolute differences between Y and
                                 *----the individual RGB components
                                 */
                                ADF_I32 diffR = R - Y;
                                ADF_I32 diffG = G - Y;
                                ADF_I32 diffB = B - Y;
                                ADF_I32 absDiffR = (diffR < 0) ? -diffR : diffR;
                                ADF_I32 absDiffG = (diffG < 0) ? -diffG : diffG;
                                ADF_I32 absDiffB = (diffB < 0) ? -diffB : diffB;

                                    
                                /**
                                 *----Sum the component differences to obtain a
                                 *----single color difference value
                                 */
                                ADF_I32 colorDifferenceValue = absDiffR +
                                absDiffG + absDiffB;

                                    
                                /**
                                 *----If the color difference value is above the
                                 *----specified threshold, desaturate the pixel by
                                 *----moving each component towards the weighted
                                 *----average Y. The factor (colorThreshold /
                                 *----colorDifferenceValue) is an ADF_I1616 fixed
                                 *----point value in the mathematical range [0,1)
                                 *----that determines how much color to preserve. If
                                 *----the factor is 0, the pixel becomes gray (i.e.,
                                 *----no color at all). Note that the implementation
                                 *----below uses only integer arithmetic to avoid
                                 *----conversions between ADF_I1616 fixed point
                                 *----values and integer values. FixedMathNote: The
                                 *----fixed point implementation uses the lookup
                                 *----table rcpTable[] (see ADFFixedMath.h) to
                                 *----compute the ADF_I1616 fixed point reciprocal of
                                 *----the color difference value. Note that each of
                                 *----the products (diffR * colorThreshold), (diffG *
                                 *----colorThreshold), and (diffB * colorThreshold)
                                 *----is a 32-bit integer. Multiplying each product
                                 *----by rcp (e.g., (diffR * colorThreshold * rcp))
                                 *----produces a 64-bit 48.16 fixed point value. Only
                                 *----the 8-bit integer portion of this 48.16 fixed
                                 *----point value is needed (i.e., bits 23:16), so
                                 *----shift each product to the right by 16 bits and
                                 *----retain only the low 8 bits.
                                 */
                                if (colorDifferenceValue > colorThreshold) {
                                    ADF_I1616 rcp = rcpTable[colorDifferenceValue];
                                    R = (ADF_U32) ((Y + ((diffR *
                                    colorThreshold * rcp) >> 16)));
                                    G = (ADF_U32) ((Y + ((diffG *
                                    colorThreshold * rcp) >> 16)));
                                    B = (ADF_U32) ((Y + ((diffB *
                                    colorThreshold * rcp) >> 16)));
                                }
                            }


                            /**
                             *----Compute the maximum density value over all three
                             *----color samples
                             */
                            maxDensity = R;
                            if (G > maxDensity) maxDensity = G;
                            if (B > maxDensity) maxDensity = B;


                        } else {


                            /**
                             *----The three color components have the same density
                             *----values. Skip color reduction (because it has no
                             *----effect) and set the maximum density value to the
                             *----red component.
                             */
                            maxDensity = R;
                        }
                        
                        
                        /**
                         *----Pack the three color components and the maximum density
                         *----value into a 32-bit word and write the result to the
                         *----density image
                         */
                        *pPixel++ = ADF_PACK_RGBA(R,G,B,maxDensity);
                    }

                        
                    /**
                     *----Initialize the density value of the red component to 255
                     *----(i.e., maximum density), which represents the case where
                     *----the red sample point is inside the shape and outside the
                     *----filter radius
                     */
                    R = 255;


                    /**
                     *----Map the red distance value to a density value using the
                     *----same method described above for the green and blue
                     *----components
                     */
                    if (distR <= minDist) R = 0;
                    else if (distR < maxDist) {


                        /**
                         *----Normalize the red distance value to an ADF_U0032 fixed
                         *----point value that lies in the mathematical range (0, 1).
                         */
                        ADF_U0032 f = ((ADF_U0032) (distR * fScale)) + fOffset;
                        

                        /**
                         *----Determine if gamma is equal to 1
                         */
                        if (gamma == I1616_CONST_1) {

                            
                            /**
                             *----Gamma is 1, so the gamma correction step can be
                             *----skipped. Map the normalized ADF_U0032 fixed point
                             *----distance value to an integer density value in the
                             *----range [0, 255].
                             */
                            R = U0032_TO_U8(f);

                            
                        } else {

                            
                            /**
                             *----Gamma is not equal to 1. Convert f from an
                             *----ADF_U0032 fixed point value to an ADF_I1616 fixed
                             *----point value by shifting f to the right by 16 bits,
                             *----perform the requested gamma correction, and map the
                             *----normalized and gamma-corrected ADF_I1616 fixed
                             *----point distance value to an integer density value in
                             *----the range [0, 255].
                             */
                            ADF_I1616 d = I1616_POW01(f >> 16, gamma);
                            R = I1616_TO_U8(d);
                        }
                    }
                }
            }
            break;
        }


        /**
         *----Horizontal striping - subpixels are horizontal and arranged vertically
         *----RGBh rendering: map distances to density values, apply color reduction
         *----if enabled, and pack the final density values and a derived alpha value
         *----into each density image pixel
         *----BGRh rendering: same as RGBv except R and B are swapped
         *----interprete R->B, B->R everywhere in this case 
         *----color reduction is not currently supported for BGRh
         */
        case ADF_REND_MODE_RGBh:
        case ADF_REND_MODE_BGRh:
        {
            ADF_U32   *pPixel = (ADF_U32 *) image->base;
            ADF_I1616 *pDistG = dist->base; /* first green row */
            ADF_I1616 *pDistR = pDistG+w;   /* first red row   */
            ADF_I1616 *pDistB = dist->base; /* becomes first blue row after j=0 */
            ADF_U32   maxDensity, tworows;
                
                
            /**
             *----Disable color reduction for ADF_REND_MODE_BGRh since unsupported
             */
            ADF_U32 useColorReduction = 0;


            /**
             *----Convert the user-specified color reduction amount to a color
             *----threshold as described in the section for RGBv above.
             */
            FS_CONST ADF_I32 colorThreshold = (I1616_CONST_1 - FLOAT_TO_I1616(
            renderGlyphData->colorReductionAmt)) >> 10;


            /**
             *----Distance buffer pointer increment between two rows.
             *----This is used to skip from one color row to the next
             */
            tworows = w<<1;


            /**
             *----Process the distance buffer from bottom to top, left to right
             */
            for (j = 0; j < h; j++) {
                ADF_I1616 distG; /* green distance for i,j */
                ADF_I1616 distR; /* red   distance for i,j */
                ADF_I1616 distB; /* blue  distance for i,j */


                /**
                 *----For LCD rendering with horizontal RGB stripes, the green
                 *----component of density image pixel (i, j) is determined from the
                 *----distance buffer sample indexed by (i, 3j), and the red and blue
                 *----components are indexed by (i, 3j+1) and (i, 3j-1),
                 *----respectively. Therefore, the blue component of each bottommost
                 *----pixel in the density image, which falls outside the density
                 *----buffer, is set to zero density and the topmost samples of the
                 *----distance buffer (which also correspond to blue components) are
                 *----computed but not used.
                 */
                ADF_U32 G, R, B;
                for (i = 0; i < w; i++) {


                    /**
                     *----Extract the green, blue, and red distance values from the
                     *----distance buffer. Note that the blue distance value (i.e.,
                     *----distB) for the first row is outside the image and the
                     *----corresponding density value (i.e. B) is mapped to zero.
                     */
                    distG = *pDistG++;      /* column i, row j   */
                    distR = *pDistR++;      /* column i, row j+1 */
                    if(j==0)
                        distB = minDist;
                    else
                        distB = *pDistB++;  /* column i, row j-1 */


                    /**
                     *----Three cases are possible: (1) the red, green, and blue
                     *----distance values are all less than the minimum distance, (2)
                     *----the red, green, and blue distance values are all greater
                     *----than the maximum distance, or (3) at least one of the red,
                     *----green, or blue distance values is in the range
                     *----[minDist,maxDist). Since the first two cases are common and
                     *----are unaffected by color reduction, handle them separately
                     *----to improve performance. 
                     */
                    if ((distR <= minDist) && (distG <= minDist) && (distB <= minDist)) {


                        /**
                         *----The distances associated with all three sample points
                         *----are all less than the minimum distance, so the density
                         *----value assigned to each sample point is zero. Skip color
                         *----reduction because it has no effect on a pixel whose RGB
                         *----values are equal. Each component of each pixel of the
                         *----density image is cleared to zero in RenderGlyph(),
                         *----so it suffices to advance the density image pointer to
                         *----the next pixel.
                         */
                        pPixel++;

                            
                    } else if ((distR >= maxDist) && 
                               (distG >= maxDist) &&
                               (distB >= maxDist)) {


                        /**
                         *----The distances associated with all three sample points
                         *----are all greater than the maximum distance, so the
                         *----density value assigned to each sample point is 255.
                         *----Skip color reduction because it has no effect on a
                         *----pixel whose RGB values are equal. The maximum density
                         *----value is also 255, so set the pixel value to RGBA =
                         *----(255,255,255,255) and advance the pointer to the next
                         *----pixel.
                         */
                        *pPixel++ = 0xFFFFFFFF;

                            
                    } else {


                        /**
                         *----At least one of the distances associated with the red,
                         *----green, or blue sample points is in the range
                         *----[minDist,maxDist). Initialize the density values of the
                         *----green and red components to 255 (i.e., maximum
                         *----density), which represents the case where the green and
                         *----red sample points are inside the shape and outside the
                         *----filter radius.
                         */
                        R = 255;
                        G = 255;
                        B = 255;

                        /**
                         *----Map the green distance value to a density value. If the
                         *----distance value is less than or equal to the minimum
                         *----distance, then the corresponding sample point is
                         *----outside the shape and outside the filter radius, so set
                         *----its density value to 0 (i.e., no density). Otherwise,
                         *----if the distance is within the filter radius, apply the
                         *----requested gamma correction and map the distance to a
                         *----density value using the precomputed fScale and fOffset
                         *----parameters.
                         */
                        if (distG <= minDist) G = 0;
                        else if (distG < maxDist) {
                            /**
                             *----Normalize the green distance value to an ADF_U0032
                             *----fixed point value that lies in the mathematical
                             *----range (0, 1).
                             */
                            ADF_U0032 f = ((ADF_U0032) (distG * fScale)) + fOffset;


                            /**
                             *----Determine if gamma is equal to 1
                             */
                            if (gamma == I1616_CONST_1) {

                        
                                /**
                                 *----Gamma is 1, so the gamma correction step can be
                                 *----skipped. Map the normalized ADF_U0032 fixed
                                 *----point distance value to an integer density
                                 *----value in the range [0, 255].
                                 */
                                G = U0032_TO_U8(f);


                            } else {
                                /**
                                 *----Gamma is not equal to 1. Convert f from an
                                 *----ADF_U0032 fixed point value to an ADF_I1616
                                 *----fixed point value by shifting f to the right by
                                 *----16 bits, perform the requested gamma
                                 *----correction, and map the normalized and
                                 *----gamma-corrected ADF_I1616 fixed point distance
                                 *----value to an integer density value in the range
                                 *----[0, 255].
                                 */
                                ADF_I1616 d = I1616_POW01(f >> 16, gamma);
                                G = I1616_TO_U8(d);
                            }
                        }
                        /**
                         *----Map the red distance value to a density value using
                         *----the same method described above for the green component
                         */
                        if (distR <= minDist) R = 0;
                        else if (distR < maxDist) {
                            /**
                             *----Normalize the blue distance value to an ADF_U0032
                             *----fixed point value that lies in the mathematical
                             *----range (0, 1).
                             */
                            ADF_U0032 f = ((ADF_U0032) (distR * fScale)) + fOffset;
                            

                            /**
                             *----Determine if gamma is equal to 1
                             */
                            if (gamma == I1616_CONST_1) {
                                /**
                                 *----Gamma is 1, so the gamma correction step can be
                                 *----skipped. Map the normalized ADF_U0032 fixed
                                 *----point distance value to an integer density
                                 *----value in the range [0, 255].
                                 */
                                R = U0032_TO_U8(f);
                                

                            } else {
                                /**
                                 *----Gamma is not equal to 1. Convert f from an
                                 *----ADF_U0032 fixed point value to an ADF_I1616
                                 *----fixed point value by shifting f to the right by
                                 *----16 bits, perform the requested gamma
                                 *----correction, and map the normalized and
                                 *----gamma-corrected ADF_I1616 fixed point distance
                                 *----value to an integer density value in the range
                                 *----[0, 255].
                                 */
                                ADF_I1616 d = I1616_POW01(f >> 16, gamma);
                                R = I1616_TO_U8(d);
                            }
                        }
                        /**
                         *----Map the blue distance value to a density value using
                         *----the same method described above for the green component
                         */
                        if (distB <= minDist) B = 0;
                        else if (distB < maxDist) {


                            /**
                             *----Normalize the blue distance value to an ADF_U0032
                             *----fixed point value that lies in the mathematical
                             *----range (0, 1).
                             */
                            ADF_U0032 f = ((ADF_U0032) (distB * fScale)) + fOffset;
                            

                            /**
                             *----Determine if gamma is equal to 1
                             */
                            if (gamma == I1616_CONST_1) {
                                /**
                                 *----Gamma is 1, so the gamma correction step can be
                                 *----skipped. Map the normalized ADF_U0032 fixed
                                 *----point distance value to an integer density
                                 *----value in the range [0, 255].
                                 */
                                B = U0032_TO_U8(f);
                                

                            } else {

                                /**
                                 *----Gamma is not equal to 1. Convert f from an
                                 *----ADF_U0032 fixed point value to an ADF_I1616
                                 *----fixed point value by shifting f to the right by
                                 *----16 bits, perform the requested gamma
                                 *----correction, and map the normalized and
                                 *----gamma-corrected ADF_I1616 fixed point distance
                                 *----value to an integer density value in the range
                                 *----[0, 255].
                                 */
                                ADF_I1616 d = I1616_POW01(f >> 16, gamma);
                                B = I1616_TO_U8(d);
                            }
                        }

                        /**
                         *----Determine if the red, green, and blue density values
                         *----are equal
                         */
                        if (R != G || G != B) {


                            /**
                             *----The three density values are not equal. Perform
                             *----color reduction if enabled and determine the
                             *----maximum density value.
                             */
                            if (useColorReduction) {


                                /**
                                 *----Map RGB to YUV using table lookups, but compute
                                 *----only the Y component. This is a weighted
                                 *----averaging technique that weights green the
                                 *----highest and blue the least.
                                 */
                                ADF_I32 Y = yuvRedTable[R] + yuvGreenTable[G] +
                                yuvBlueTable[B];


                                /**
                                 *----Compute the absolute differences between Y and
                                 *----the individual RGB components
                                 */
                                ADF_I32 diffR = R - Y;
                                ADF_I32 diffG = G - Y;
                                ADF_I32 diffB = B - Y;
                                ADF_I32 absDiffR = (diffR < 0) ? -diffR : diffR;
                                ADF_I32 absDiffG = (diffG < 0) ? -diffG : diffG;
                                ADF_I32 absDiffB = (diffB < 0) ? -diffB : diffB;
                                    
                                /**
                                 *----Sum the component differences to obtain a
                                 *----single color difference value
                                 */
                                ADF_I32 colorDifferenceValue = absDiffR +
                                absDiffG + absDiffB;
                                    
                                /**
                                 *----If the color difference value is above the
                                 *----specified threshold, desaturate the pixel by
                                 *----moving each component towards the weighted
                                 *----average Y. See comments on color reduction
                                 *----for the case of RGBv above.
                                 */
                                if (colorDifferenceValue > colorThreshold) {
                                    ADF_I1616 rcp = rcpTable[colorDifferenceValue];
                                    R = (ADF_U32) ((Y + ((diffR *
                                    colorThreshold * rcp) >> 16)));
                                    G = (ADF_U32) ((Y + ((diffG *
                                    colorThreshold * rcp) >> 16)));
                                    B = (ADF_U32) ((Y + ((diffB *
                                    colorThreshold * rcp) >> 16)));
                                }
                            }


                            /**
                             *----Compute the maximum density value over all three
                             *----color samples
                             */
                            maxDensity = R;
                            if (G > maxDensity) maxDensity = G;
                            if (B > maxDensity) maxDensity = B;


                        } else {


                            /**
                             *----The three color components have the same density
                             *----values. Skip color reduction (because it has no
                             *----effect) and set the maximum density value to the
                             *----red component.
                             */
                            maxDensity = R;
                        }
                        
                        
                        /**
                         *----Pack the three color components and the maximum density
                         *----value into a 32-bit word and write the result to the
                         *----density image
                         */
                        *pPixel++ = ADF_PACK_RGBA(R,G,B,maxDensity);
                    }

                } /* for i = 0; i < w; */

                /**
                 *----The distance buffer pointers now point to the next subpixel row, i.e.
                 *----pDistG points to the next red row, 
                 *----pDistR points to the next blue row
                 *----pDistB points to the next green row
                 *----Skip two rows each so that they point to their respective subpixels
                 */
                pDistG += tworows;
                pDistR += tworows;
                pDistB += tworows;

            } /* for j = 0; j < h; */
            break;
        }

            
        /**
         *----An invalid display mode
         */
        default:
        {
            break;
        }
    }
}


/**
 *-----------------------------------------------------------------------------------
 *    CLEARING AN ADF IMAGE
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Clear the specified ADFImage by setting each pixel or pixel component to zero
 *-----------------------------------------------------------------------------------
 */
static ADF_Void ClearADFImage (ADFImage *image)
{
    /**
     *----Process single channel and RGBA density images separately
     */
    if (image->type == ADF_IMAGE_TYPE_GRAY) {


        /**
         *----Set each image pixel to zero
         */
        SYS_MEMSET(image->base, 0, image->w * image->h * sizeof(ADF_U8));


    } else {

        
        /**
         *----Set each image pixel's RGBA values to zero
         */
        SYS_MEMSET(image->base, 0, image->w * image->h * sizeof(ADF_U32));
    }
}


/**
 *-------------------------------------------------------------------------------
 *    END: FIXED POINT MATH ONLY
 *-------------------------------------------------------------------------------
 */


/**
 *-------------------------------------------------------------------------------
 *    END: IMPLICIT ADFS ONLY
 *-------------------------------------------------------------------------------
 */
#endif

/**
 *-----------------------------------------------------------------------------------
 *    END: iType Edge Rendering
 *-----------------------------------------------------------------------------------
 */
#endif
